/* 
 * Copyright (C) 1998 Albrecht Gebhardt
 *

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 * This is a library of functions for QIC volume table modification.
 * The functions are taken from the vtblc utility  distributed with the 
 * Linux ftape driver version 3.04d.
 * 
 * The original Copyright notice follows:
 */

/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *     This program is a small utility to manipulate certain fields of
 *     the volume table of a floppy tape cartridge.  To be used with
 *     the QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

char src[] = "$Source: /usr/local/cvsroot/libvtblc/vtblc.c,v $";
char rev[] = "$Revision: 1.4 $";
char dat[] = "$Date: 1998/03/16 05:41:10 $";


#include "vtblc.h"


void set_vtbl_cntl(int verbose)
{
	volume_cntl.verbose = verbose;
}

void get_vtbl_cntl(char *errstr, char *drivetype)
{
	strcpy(errstr, volume_cntl.errstr);
	strcpy(drivetype, volume_cntl.drivetype);
}

/*
 *  open the tape device, try to determine whether it is a floppy tape,
 *  and store the hardware status in drive_config, resp. tape_config.
 *  mmap() the dma buffers if op_mode != PROBING
 *  
 *  Returns the tape fd on success, -1 otherwise
 *
 */

int raw_tape_open(const char *name, int mode)
{
	int tape_fd;
	struct mtget mtget;
	static vendor_struct vendors[] = QIC117_VENDORS;
	vendor_struct *vendor;

	/* open the tape device
	 */
	if ((tape_fd = open(name, mode)) == -1) {
		sprintf(volume_cntl.errstr, 
				"Error opening raw tape device: %s", 
				strerror(errno));
		return -1;
	}
	/* get its status
	 */
	if (ioctl(tape_fd, MTIOCGET, &mtget) == -1) {
		sprintf(volume_cntl.errstr, 
				"Error getting tape drive status: %$", strerror(errno));
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_DR_OPEN(mtget.mt_gstat)) {
		sprintf(volume_cntl.errstr, "Error: No tape cartridge present!");
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_WR_PROT(mtget.mt_gstat)) {
		sprintf(volume_cntl.errstr, "Warning: Write protected cartridge!");
	}
	if (!GMT_ONLINE(mtget.mt_gstat)) {
		sprintf(volume_cntl.errstr, "Error: Tape drive is offline!");
		(void)close(tape_fd);
		return -1;
	}
	if ((mtget.mt_type & MT_ISFTAPE_FLAG) == 0) {
		sprintf(volume_cntl.errstr, "Error: This is not a floppy tape drive!");
		(void)close(tape_fd);
		return -1;
	}
	mtget.mt_type &= ~MT_ISFTAPE_FLAG; /* clear the flag bit */
	vendor = &vendors[0];
	while (vendor->vendor_id != UNKNOWN_VENDOR &&
		   vendor->vendor_id != mtget.mt_type) {
		vendor++;
	}
	sprintf(volume_cntl.drivetype, 
			"Tape drive type: %s (0x%04lx)\n", 
			vendor->name, mtget.mt_type);

	return tape_fd;
}

int raw_tape_close(int tape_fd)
{
	const struct mtop rewind = { MTREW, 1 };
	int result = 0;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		sprintf(volume_cntl.errstr, 
				"Ioctl error rewinding tape: %s\n", strerror(errno));
		result = -1;
	}
	if (close(tape_fd) == -1) {
		sprintf(volume_cntl.errstr, 
				"Error closing tape device: %s\n", strerror(errno));
		result = -1;
	}
	return result;
}

int read_vtbl(int tape_fd, vtbl *volumes, u_int8_t *buffer,
			  int *first_data_segment, int *last_data_segment)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int vtbl_cnt; 
	int end_seg = 0;
	int fmt_code;
	const char *ids[] = VTBL_IDS;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		sprintf(volume_cntl.errstr, 
				"Ioctl error rewinding tape: %s\n", strerror(errno));
		return -1;
	}
	if ((fmt_code = read_header_segment(tape_fd,
										first_data_segment,
										last_data_segment)) == -1) {
		return -1;
	}
	if (volume_cntl.verbose) {
		printf("Reading volume table segment ... ");
		fflush(stdout);
	}
	memset(&ft_seg, 0, sizeof(ft_seg));
	ft_seg.mt_data  = buffer;
	ft_seg.mt_segno = *first_data_segment;
	ft_seg.mt_mode  = MT_FT_RD_SINGLE;
	if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
		sprintf(volume_cntl.errstr, "Ioctl error reading volume table: %s\n",
				strerror(errno));
		return -1;
	}
	if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
		sprintf(volume_cntl.errstr, "Short read() reading volume table: %d\n",
				ft_seg.mt_result);
		return -1;
	}
	if (volume_cntl.verbose) {
		printf("done.\n");
	}
	vtbl_cnt = 0;
	if (fmt_code == fmt_big) {
		end_seg = *first_data_segment;
	}
	while (!memcmp(buffer, ids[0], 4) ||
		   !memcmp(buffer, ids[1], 4) ||
		   !memcmp(buffer, ids[2], 4) ||
		   !memcmp(buffer, ids[3], 4)) {
		memcpy(volumes[vtbl_cnt].id, buffer, 4);
		if (!memcmp(buffer, "VTBL", 4)) {
			memcpy(&volumes[vtbl_cnt].date, buffer + VTBL_DATE, 4);
			memcpy(volumes[vtbl_cnt].label, buffer + VTBL_DESC, 44);
			volumes[vtbl_cnt].label[44] = '\0';
			if (fmt_code == fmt_big) {
				volumes[vtbl_cnt].start = end_seg + 1;
				memcpy(&volumes[vtbl_cnt].end, buffer + VTBL_SCSI_SEGS, 4);
				volumes[vtbl_cnt].end += end_seg;
				end_seg = volumes[vtbl_cnt].end;
			} else {
				memcpy(&volumes[vtbl_cnt].start, buffer + VTBL_START, 2);
				memcpy(&volumes[vtbl_cnt].end,   buffer + VTBL_END,   2);
				volumes[vtbl_cnt].start &= 0x0000ffff;
				volumes[vtbl_cnt].end   &= 0x0000ffff;
			}
			volumes[vtbl_cnt].fmt_code = fmt_code;
		}
		vtbl_cnt ++;
		buffer += VTBL_SIZE;
	}
	return vtbl_cnt;
}

/*  returns the max. number of segments
 */
int read_header_segment(int tape_fd,
						int *first_data_segment, int *last_data_segment)
{
		struct mtftseg ft_seg;
		int i;
		const unsigned long hseg_magic = FT_HSEG_MAGIC;
		u_int8_t hseg[FT_SEGMENT_SIZE];
		
		if (volume_cntl.verbose) {
			printf("Reading header segment ... ");
			fflush(stdout);
		}
		for (i = 0; i < 64; i++) {
			memset(&ft_seg, 0, sizeof(ft_seg));
			ft_seg.mt_data  = hseg;
			ft_seg.mt_segno = i;
			ft_seg.mt_mode  = MT_FT_RD_AHEAD;
			if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
				sprintf(volume_cntl.errstr, "Ioctl error reading header segment: %s\n",
						strerror(errno));
				return -1;
			}
			if (ft_seg.mt_result == FT_SEGMENT_SIZE) {
				break;
			}
		}
		if (memcmp(&hseg_magic, hseg, 4)) {
			sprintf(volume_cntl.errstr, "Unable to read the header segment\n");
			return -1;
		}
		if (volume_cntl.verbose) {
			printf("done.\n");
		}
		if (hseg[FT_FMT_CODE] == fmt_big) {
			memcpy(first_data_segment, &hseg[FT_6_FRST_SEG], 4);
			memcpy(last_data_segment,  &hseg[FT_6_LAST_SEG], 4);
		} else {
			memcpy(first_data_segment, &hseg[FT_FRST_SEG], 2);
			memcpy(last_data_segment,  &hseg[FT_LAST_SEG], 2);
			*first_data_segment &= 0x0000ffff;
			*last_data_segment &= 0x0000ffff;
		}
		return (int)hseg[FT_FMT_CODE];
}

void print_vtbl(const vtbl *volumes, int maxnum,
				int first_seg, int last_seg)
{
	int i;

	printf("%3s %3s %*s %*s %*s %8s %8s\n",
		   "Nr", "Id", 14, "Label", 22, "Date", 15, "Start", "End", "Space");
	for (i = 0; i < 80; i++) {
		printf("-");
	}
	printf("\n");
	for (i = 0; i < maxnum; i++) {
		print_one_vtbl(volumes, i, first_seg, last_seg);
	}
}

void print_long_vtbl(const vtbl *volumes, int maxnum,
					 int first_seg, int last_seg)
{
	int i;

	printf("%3s %3s %*s %*s \n",
		   "Nr", "Id", 14, "Label", 44, "Date");
	for (i = 0; i < 80; i++) {
		printf("-");
	}
	printf("\n");
	for (i = 0; i < maxnum; i++) {
		print_one_long_vtbl(volumes, i, first_seg, last_seg);
	}
}

void print_one_vtbl(const vtbl *volumes,
					int vtbl_no, int first_seg, int last_seg)
{
	char label[45];
	char sig[5];
	double usage;

	memcpy(label, volumes[vtbl_no].label, 44);
	label[22] = '\0';
	memcpy(sig, volumes[vtbl_no].id, 4);
	sig[4] = '\0';
	if (!strcmp(sig, "VTBL")) {
		usage = (double)(volumes[vtbl_no].end - volumes[vtbl_no].start + 1);
		usage = usage/(double)(last_seg - first_seg)*100.0;
		printf("%3d %4s \"%-*s\" %*s %8d %8d    %2.2f%%\n",
			   vtbl_no, sig, 22, label, 18, decode_date(volumes[vtbl_no].date),
			   volumes[vtbl_no].start, volumes[vtbl_no].end, usage);
	} else {
		printf("%4d %4s", vtbl_no, sig);
	}
}

void print_one_long_vtbl(const vtbl *volumes,
						 int vtbl_no, int first_seg, int last_seg)
{
	char label[45];
	char sig[5];
	int len;

	memcpy(label, volumes[vtbl_no].label, 44);
	if ((len = strlen(label)) < 4) {
		memset(label + len, ' ', 44 - len);
	}
	label[44] = '\0';
	memcpy(sig, volumes[vtbl_no].id, 4);
	sig[4] = '\0';
	if (!strcmp(sig, "VTBL")) {
		printf("%3d %4s \"%-*s\" %*s\n",
			   vtbl_no, sig, 44, label, 18, decode_date(volumes[vtbl_no].date));
	} else {
		printf("%4d %4s", vtbl_no, sig);
	}
}

char *decode_date(u_int32_t timestamp)
{
	struct tm tapetm;
	time_t tapetime;
	static char date[18];

	memset(&tapetm, 0, sizeof(tapetm));
	
	tapetm.tm_year  = timestamp >> FT_YEAR_SHIFT;
	tapetm.tm_year += FT_YEAR_0 - 1900;
	timestamp      &= FT_TIME_MASK;

	tapetm.tm_sec   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_min   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_hour  = timestamp % 24;
	timestamp      /= 24;
	tapetm.tm_mday  = timestamp % 31 + 1;
	timestamp      /= 31;
	tapetm.tm_mon   = timestamp % 12;
	tapetm.tm_isdst = -1;
	if ((tapetime = mktime(&tapetm)) == (time_t)(-1)) {
		return "invalid";
	} else {
		(void)strftime(date, 18, "%T %D", &tapetm);
		return date;
	}
}

int write_vtbl(int tape_fd, const vtbl *volumes, u_int8_t *buffer,
			   int vtbl_cnt, int first_data_segment, int modified)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int i;
	u_int8_t *p = buffer;

	for (i = 0; i < vtbl_cnt; i++) {
		if (volumes[i].modified) {
			memcpy(p + VTBL_SIG, volumes[i].id, 4);
			if (volumes[i].fmt_code == fmt_big) {
				u_int32_t segs = volumes[i].end - volumes[i].start + 1;
				memcpy(p + VTBL_SCSI_SEGS, &segs, 4);
			} else {
				memcpy(p + VTBL_START, &volumes[i].start, 2);
				memcpy(p + VTBL_END, &volumes[i].end, 2);
			}
			memcpy(p + VTBL_DESC, volumes[i].label, 44);
			memcpy(p + VTBL_DATE, &volumes[i].date, 4);
			modified ++;
		}
		p += VTBL_SIZE;
	}
	memset(p, 0, VTBL_SIZE * (MAX_VOLUMES - vtbl_cnt));
	if (modified) {
		if (volume_cntl.verbose) {
			printf("Writing volume table segment ... ");
			fflush(stdout);
		}
		memset(&ft_seg, 0, sizeof(ft_seg));
		ft_seg.mt_data  = buffer;
		ft_seg.mt_segno = first_data_segment;
		ft_seg.mt_mode  = MT_FT_WR_SINGLE;
		if (ioctl(tape_fd, MTIOCWRFTSEG, &ft_seg) == -1) {
			sprintf(volume_cntl.errstr, "Ioctl error writing volume table: %s\n",
					strerror(errno));
			return -1;
		}
		if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
			sprintf(volume_cntl.errstr, "Short write() reading volume table: %d\n",
					ft_seg.mt_result);
			return -1;
		}
		if (volume_cntl.verbose) {
			printf("done.\n");
		}
	}
	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		sprintf(volume_cntl.errstr, "Ioctl error rewinding tape: %s\n", strerror(errno));
		return -1;
	}
	return 0;
}

int set_date(/*const*/ char *date,
			 vtbl *volumes, int maxnum, int vtbl_no)
{
	time_t raw_time;
	struct tm tapetime;
	struct tm *tapetimep = &tapetime;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		sprintf(volume_cntl.errstr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	if (date == NULL) {
		time(&raw_time);
		tapetimep = localtime(&raw_time);
	} else {
		(void)strptime(date, "%T %D", tapetimep);
	}
	volumes[vtbl_no].date = FT_TIME_STAMP(tapetimep->tm_year + 1900, 
										  tapetimep->tm_mon, 
										  tapetimep->tm_mday-1,
										  tapetimep->tm_hour,
										  tapetimep->tm_min,
										  tapetimep->tm_sec);
	volumes[vtbl_no].modified = 1;
	return 0;
}

int set_label(const char *desc, vtbl *volumes, int maxnum, int vtbl_no)
{
	int len;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		sprintf(volume_cntl.errstr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	strncpy(volumes[vtbl_no].label, desc, 44);
	if ((len = strlen(desc)) < 44) {
		memset(volumes[vtbl_no].label + len, ' ', 44 - len);
	}
	volumes[vtbl_no].modified = 1;
	return 0;
}

int set_start(int start, vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		sprintf(volume_cntl.errstr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	volumes[vtbl_no].start = start;
	volumes[vtbl_no].modified = 1;
	return 0;
}

int set_end(int end, vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		sprintf(volume_cntl.errstr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	volumes[vtbl_no].end = end;
	volumes[vtbl_no].modified = 1;
	return 0;
}

int set_id(vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		sprintf(volume_cntl.errstr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	memcpy(volumes[vtbl_no].id, "VTBL", 4);
	volumes[vtbl_no].modified = 1;
	return 0;
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
 
