///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "reference_element_face_transformation.h"

namespace rheolef {

// Let S and K such that S is the i-th side of K
// then map a point hat_x defined in the reference element side hat_S
// into tilde_x defined in the reference element tilde_K
// note: used to build a quadrature formulae on a side of K

template<class T>
point_basic<T>
reference_element_face_transformation (
  reference_element             tilde_K,
  const side_information_type&  sid,
  const point_basic<T>&		hat_x)
{
  // d=1: tilde_x = tilde_a0
  // d=3: tilde_x = tilde_a0 + hat_x[0]*(tilde_a1 - tilde_a0)
  // d=3: tilde_x = tilde_a0 + hat_x[0]*(tilde_a1 - tilde_a0) + hat_x[1]*(tilde_a2 - tilde_a0)
  // where tilde_ai are the vertices of the transformed side tilde_S in tilde_K
  switch (sid.dim) {
    case 0: {
      check_macro (sid.shift == 0, "unexpected shift="<<sid.shift<<" on a0d side");
      check_macro (sid.orient > 0, "unexpected orient="<<sid.orient<<" on a 0d side");
      size_t i0 = tilde_K.subgeo_local_vertex (sid.dim, sid.loc_isid, 0);
      const point& tilde_a0 = tilde_K.vertex (i0);
      return tilde_a0;
    }
    case 1: {
      check_macro (sid.shift == 0, "unexpected shift="<<sid.shift<<" on an edge");
      size_t i0 = tilde_K.subgeo_local_vertex (sid.dim, sid.loc_isid, 0),
             i1 = tilde_K.subgeo_local_vertex (sid.dim, sid.loc_isid, 1);
      if (sid.orient < 0) std::swap (i0, i1);
      const point& tilde_a0 = tilde_K.vertex (i0);
      const point& tilde_a1 = tilde_K.vertex (i1);
      return tilde_a0 + hat_x[0]*(tilde_a1 - tilde_a0);
    }
    case 2: {
      size_t nv = sid.n_vertex;
      size_t i0 = tilde_K.subgeo_local_vertex (sid.dim, sid.loc_isid, sid.shift%nv),
             i1 = tilde_K.subgeo_local_vertex (sid.dim, sid.loc_isid, (sid.shift+1)%nv),
             i2 = tilde_K.subgeo_local_vertex (sid.dim, sid.loc_isid, (sid.shift+nv-1)%nv);
      if (sid.orient < 0) std::swap (i1, i2);
      const point& tilde_a0 = tilde_K.vertex (i0);
      const point& tilde_a1 = tilde_K.vertex (i1);
      const point& tilde_a2 = tilde_K.vertex (i2);
      return tilde_a0 + hat_x[0]*(tilde_a1 - tilde_a0)
                      + hat_x[1]*(tilde_a2 - tilde_a0);
    }
    default: {
	fatal_macro ("invalid subgeo_dim="<<sid.dim);
	return point_basic<T>();
    }
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T) 				\
template							\
point_basic<T>							\
reference_element_face_transformation (				\
  reference_element             tilde_K,			\
  const side_information_type&  sid,				\
  const point_basic<T>&		hat_x);

_RHEOLEF_instanciation(Float)

} // namespace rheolef
