------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2006 Maxim Reznik.                                        --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-object.adb,v 1.4 2008/07/03 06:22:48 vagul Exp $

with OCI.Thread;

with Ada.Unchecked_Conversion;
with System.Address_To_Access_Conversions;

package body OCI.Thick.Object is

   package Convert is new System.Address_To_Access_Conversions (Object_Node);

   function Get_TDO (Connect : in Connections.Connection) return OCIType;

   protected TDO_Lock is
      entry Get;
      procedure Release;
   private
      Open : Boolean := True;
   end TDO_Lock;

   TDO       : aliased OCIType;
   TDO_Ready : Boolean := False;
   pragma Atomic (TDO_Ready);

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Stmt     : in     Statements.Statement;
      Item     : in out Variable;
      Position : in     Positive)
   is
      RC  : SWord;
      TDO : constant OCIType := Get_TDO (Statements.Get_Connection (Stmt));
   begin
      RC := OCIBindByPos
        (Stmtp    => OCIStmt (Handle (Stmt)),
         Bindpp   => Item.Bind'Access,
         Errhp    => Thread.Error,
         Position => Ub4 (Position),
         Valuep   => Empty_Handle,
         Value_Sz => 0,
         Dty      => SQLT_NTY,
         Indp     => null);

      Check_Error (RC);

      RC := OCIBindObject
        (Bindp    => Item.Bind,
         Errhp    => Thread.Error,
         TDO      => TDO,
         Value    => Item.Data'Address,
         Value_Sz => null,
         Indp     => Item.Ind'Address,
         Ind_Sz   => null);

      Check_Error (RC);
   end Bind;

   procedure Bind
     (Stmt     : in     Statements.Statement;
      Item     : in out Variable;
      Name     : in     String)
   is
      RC  : SWord;
      TDO : constant OCIType := Get_TDO (Statements.Get_Connection (Stmt));
   begin
      RC := OCIBindByName
        (Stmtp       => OCIStmt (Handle (Stmt)),
         Bindpp      => Item.Bind'Access,
         Errhp       => Thread.Error,
         Placeholder => C.To_C (Name),
         Placeh_Len  => Name'Length,
         Valuep      => Empty_Handle,
         Value_Sz    => 0,
         Dty         => SQLT_NTY,
         Indp        => null);

      Check_Error (RC);

      RC := OCIBindObject
        (Bindp    => Item.Bind,
         Errhp    => Thread.Error,
         TDO      => TDO,
         Value    => Item.Data'Address,
         Value_Sz => null,
         Indp     => Item.Ind'Address,
         Ind_Sz   => null);

      Check_Error (RC);
   end Bind;

   ------------
   -- Create --
   ------------

   procedure Create
     (Connect : in     Connections.Connection;
      Item    :    out Variable)
   is
      TDO : constant OCIType := Get_TDO (Connect);
      RC  : SWord;
   begin
      RC := OCIObjectNew
        (Thread.Environment,
         Thread.Error,
         OCISvcCtx (Handle (Connect)),
         OCI_TYPECODE_OBJECT,
         TDO,
         Empty_Handle,
         OCI_DURATION_SESSION,
         0,
         Item.Data'Access);

      Check_Error (RC);

      RC := OCIObjectGetInd
        (Thread.Environment,
         Thread.Error,
         Item.Data,
         Item.Ind'Access);

      Check_Error (RC);
   end Create;

   ------------
   -- Define --
   ------------

   procedure Define
     (Stmt     : in     Statements.Statement;
      Item     : in out Variable;
      Position : in     Positive)
   is
      RC  : SWord;
      TDO : constant OCIType := Get_TDO (Statements.Get_Connection (Stmt));
   begin
      RC := OCIDefineByPos
        (Stmtp    => OCIStmt (Handle (Stmt)),
         Defnpp   => Item.Define'Access,
         Errhp    => Thread.Error,
         Position => Ub4 (Position),
         Value    => Empty_Handle,
         Value_Sz => 0,
         Dty      => SQLT_NTY,
         Indp     => null);

      Check_Error (RC);

      RC := OCIDefineObject
        (Defnp    => Item.Define,
         Errhp    => Thread.Error,
         TDO      => TDO,
         Value    => Item.Data'Address,
         Value_Sz => null,
         Indp     => Item.Ind'Address,
         Ind_Sz   => null);

      Check_Error (RC);
   end Define;

   ----------
   -- Free --
   ----------

   procedure Free  (Item : in out Variable) is
      RC : constant SWord
        := OCIObjectFree (Thread.Environment, Thread.Error, Item.Data, 0);
   begin
      Check_Error (RC);

      Item.Data := Empty_Handle;
      Item.Ind  := Empty_Handle;
   end Free;

   -------------------
   -- Get_Indicator --
   -------------------

   function Get_Indicator (Item : Variable) return Indicator_Access is
      function "+" is
         new Ada.Unchecked_Conversion (OCIHandle, Indicator_Access);
   begin
      return +Item.Ind;
   end Get_Indicator;

   function Get_Indicator (Item : Variable) return Indicator is
   begin
      return Get_Indicator (Item).all;
   end Get_Indicator;

   -------------
   -- Get_TDO --
   -------------

   function Get_TDO (Connect : in Connections.Connection) return OCIType is
      use type C.int;
      use type OCIType;
      RC    : SWord;
      Empty : Text (1 .. 0);
      Name  : constant Text := C.To_C (Type_Name, False);
   begin
      if not TDO_Ready then
         TDO_Lock.Get;
         RC := OCI_SUCCESS;

         if not TDO_Ready then
            RC := OCITypeByName
              (Thread.Environment,
               Thread.Error,
               OCISvcCtx (Handle (Connect)),
               Empty,
               0,
               Name,
               Type_Name'Length,
               Empty,
               0,
               OCI_DURATION_SESSION,
               OCI_TYPEGET_ALL,
               TDO'Access);

            TDO_Ready := RC = OCI_SUCCESS;
         end if;

         TDO_Lock.Release;

         Check_Error (RC);
      end if;

      return TDO;
   end Get_TDO;

   ---------------
   -- Get_Value --
   ---------------

   function Get_Value (Item : Variable) return Object_Access is
   begin
      return Object_Access (Convert.To_Pointer (Item.Data));
   end Get_Value;

   function Get_Value (Item : Variable) return Object_Node is
   begin
      return Get_Value (Item).all;
   end Get_Value;

   -------------------
   -- Set_Indicator --
   -------------------

   procedure Set_Indicator
     (Item    : in out Variable;
      Nullity : in     Indicator) is
   begin
      Get_Indicator (Item).all := Nullity;
   end Set_Indicator;

   ---------------
   -- Set_Value --
   ---------------

   procedure Set_Value
     (Item  : in out Variable;
      Value : in     Object_Node) is
   begin
      Get_Value (Item).all := Value;
   end Set_Value;

   --------------
   -- TDO_Lock --
   --------------

   protected body TDO_Lock is

      entry Get when Open is
      begin
         Open := False;
      end Get;

      procedure Release is
      begin
         Open := True;
      end Release;
   end TDO_Lock;

end OCI.Thick.Object;
