# Copyright (c) 2022 by Osyotr
# https://gist.github.com/Osyotr/e2d415312a10183bc7c614013494bf21
# Boost Software License
# Permission is hereby granted, free of charge, to any person or organization
# obtaining a copy of the software and accompanying documentation covered by
# this license (the "Software") to use, reproduce, display, distribute,
# execute, and transmit the Software, and to prepare derivative works of the
# Software, and to permit third-parties to whom the Software is furnished to
# do so, all subject to the following:
#
# The copyright notices in the Software and this entire statement, including
# the above license grant, this restriction and the following disclaimer,
# must be included in all copies of the Software, in whole or in part, and
# all derivative works of the Software, unless such copies or derivative
# works are solely in the form of machine-executable object code generated by
# a source language processor.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
# SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
# FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
# ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

include_guard(GLOBAL)

function(x_vcpkg_bootstrap)
    if(NOT GIT_EXECUTABLE)
        find_package(Git REQUIRED)
    endif()

    set(X_VCPKG_GIT_REPOSITORY_URL "https://github.com/Microsoft/vcpkg" CACHE STRING "Vcpkg git repository")
	set(X_VCPKG_CLONE_DIR "${CMAKE_SOURCE_DIR}/vcpkg" CACHE PATH "Vcpkg clone directory")

    if(NOT EXISTS "${X_VCPKG_CLONE_DIR}/.git")
        message(STATUS "Cloning vcpkg into ${X_VCPKG_CLONE_DIR}")
        execute_process(
            COMMAND "${GIT_EXECUTABLE}" clone --quiet "${X_VCPKG_GIT_REPOSITORY_URL}" "${X_VCPKG_CLONE_DIR}"
            ERROR_VARIABLE _vcpkg_git_clone_error
        )
        if(_vcpkg_git_clone_error)
            message(FATAL_ERROR "Could not clone vcpkg repository from ${X_VCPKG_GIT_REPOSITORY_URL}\nMake sure you have access rights and the URL is valid.")
        endif()
        message(STATUS "Cloning vcpkg into ${X_VCPKG_CLONE_DIR} - done")
    endif()

    execute_process(
        COMMAND "${GIT_EXECUTABLE}" rev-parse HEAD
        WORKING_DIRECTORY "${X_VCPKG_CLONE_DIR}"
        OUTPUT_VARIABLE _vcpkg_current_head_sha
        OUTPUT_STRIP_TRAILING_WHITESPACE
        COMMAND_ERROR_IS_FATAL ANY
    )
    set(_baseline_sha "${_vcpkg_current_head_sha}")

    file(READ "${CMAKE_SOURCE_DIR}/vcpkg-configuration.json" _vcpkg_configuration_json_contents)
    string(JSON _baseline_sha ERROR_VARIABLE _get_baseline_err GET "${_vcpkg_configuration_json_contents}" "default-registry" "baseline")

    set(_should_update_vcpkg FALSE)
    if(NOT "${_vcpkg_current_head_sha}" STREQUAL "${_baseline_sha}")
        set(_should_update_vcpkg TRUE)
    endif()

    if(_should_update_vcpkg)
        message(STATUS "Fetching changes from vcpkg upstream")
        execute_process(
            COMMAND "${GIT_EXECUTABLE}" fetch --quiet --prune
            WORKING_DIRECTORY "${X_VCPKG_CLONE_DIR}"
            # No error checking here to allow offline usage
        )
        message(STATUS "Fetching changes from vcpkg upstream - done")

        message(STATUS "Switching vcpkg HEAD to ${_baseline_sha}")
        execute_process(
            COMMAND "${GIT_EXECUTABLE}" reset --quiet --hard "${_baseline_sha}"
            COMMAND "${GIT_EXECUTABLE}" clean -qfd
            WORKING_DIRECTORY "${X_VCPKG_CLONE_DIR}"
            COMMAND_ERROR_IS_FATAL ANY
        )
        message(STATUS "Switching vcpkg HEAD to ${_baseline_sha} - done")

        # Remove vcpkg executable to trigger bootstrap
        if(CMAKE_HOST_SYSTEM_NAME STREQUAL "Windows")
            file(REMOVE "${X_VCPKG_CLONE_DIR}/vcpkg.exe")
        else()
            file(REMOVE "${X_VCPKG_CLONE_DIR}/vcpkg")
        endif()
    endif()
endfunction()
