#!/usr/bin/env python3
# THIS FILE IS PART OF THE CYLC SUITE ENGINE.
# Copyright (C) NIWA & British Crown (Met Office) & Contributors.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""cylc [task] remote-init [--indirect-comm=ssh] UUID RUND

(This command is for internal use.)
Install suite service files on a task remote (i.e. a [owner@]host):
    .service/contact: All task -> suite communication methods.

Content of items to install from a tar file read from STDIN.

Return:
    0:
        On success or if initialisation not required:
        - Print task_remote_cmd.REMOTE_INIT_NOT_REQUIRED if initialisation
          not required (e.g. remote has shared file system with suite host).
        - Print task_remote_cmd.REMOTE_INIT_DONE on success.
    1:
        On failure.

"""
from cylc.flow.remote import remrun


def main():
    if not remrun():
        from cylc.flow.option_parsers import CylcOptionParser as COP
        from cylc.flow.task_remote_cmd import remote_init

        parser = COP(
            __doc__,
            argdoc=[
                ("UUID", "UUID of current suite server process"),
                ("RUND", "The run directory of the suite"),
            ],
        )
        parser.add_option(
            "--indirect-comm",
            metavar="METHOD",
            type="choice",
            choices=["ssh"],
            help="specify use of indirect communication via e.g. ssh",
            action="store",
            dest="indirect_comm",
            default=None,
        )
        options, (uuid_str, rund) = parser.parse_args()
        remote_init(uuid_str, rund, indirect_comm=options.indirect_comm)


if __name__ == "__main__":
    main()
