/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

/******************************************************************************

lcmaps_gridlist.c

Description:
    Set of functions to read files in the style of the gridmapfile.
    This is basically a copy of gridmap.c from globus with some
    modifications. This was done to maintain compatibility with the 
    gridmap file.

CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-voms/src/gridlist/lcmaps_gridlist.c,v $
    $Date: 2010-02-19 06:01:37 $
    $Revision: 1.13 $
    $Author: okoeroo $

******************************************************************************/

#ifndef LCMAPS_GRIDLIST_C
#define LCMAPS_GRIDLIST_C
#endif


/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_plugins_basic_config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <pwd.h>
#include <fnmatch.h>

#if HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <lcmaps/lcmaps_modules.h>
#include "lcmaps_gridlist.h"
/******************************************************************************
                               Type definitions
******************************************************************************/

typedef struct _gridmap_line_s {
  char *dn;
  char **user_ids;
} gridmap_line_t;

/******************************************************************************
                                Definitions
******************************************************************************/
#define WHITESPACE_CHARS        " \t\n"

#define QUOTING_CHARS           "\""

#define ESCAPING_CHARS          "\\"

#define COMMENT_CHARS           "#"

#define WILD_CHARS              '*'

/* Characters seperating user ids in the gridmap file */
#define USERID_SEP_CHARS        ","

/*
 * Characters that terminate a user id in the gridmap file. This
 * is a combination of whitespace and seperators.
 */
#define USERID_TERMINATOR_CHARS     USERID_SEP_CHARS WHITESPACE_CHARS

#ifndef NUL
#define NUL             '\0'
#endif

/*
 * Number of user id slots to allocate at a time
 * Arbitraty value, but must be >= 2.
 */
#define USERID_CHUNK_SIZE       256

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

static int gridmap_default_path(char **ppath);

static int gridmap_find_dn(const char        *const dn,
                 gridmap_line_t **gline,
                 char            *globusidfile,
                 unsigned short   matching_type,
                 char *           searchstr);

static int
gridmapdir_userid(char *     globusidp,
                  char *     usernameprefix,
                  char **    useridp,
        unsigned short       matching_type);

static int gridmap_parse_line(char *line,
                gridmap_line_t **gline);

static void gridmap_free_gridmap_line(gridmap_line_t *line);

static int gridmap_parse_globusid(const char * unparse,
                        char **pparsed);

static int xdigit_to_value(char xdigit);

/* added by EDG/EGEE */
static int match_dn(
        const char *    dn,
        const char *    match,
        unsigned short  matching_type);
static int match_username(
        const char *    username,
        const char *    regex,
        unsigned short  matching_type);
static int gridmapdir_newlease(
        char *          encodedglobusidp,
        char *          usernameprefix,  
        unsigned short  matching_type);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static int
match_dn(
    const char * dn,
    const char * match,
    unsigned short matching_type
)
{
    if ( (matching_type&MATCH_WILD_CHARS) == MATCH_WILD_CHARS )
        return fnmatch(match, dn, 0);
    else
        return strcmp(dn, match);
}

/******************************************************************************
Function:   match_username
Description:
        This function checks if a username matches.
        The match is expressed in the following regex:
        - if matching_type contains the MATCH_STRICT_PREFIX_NUM flag:
                username=~/^${usernameprefix}[0-9]+/ 
        - otherwise:
                username=~/^${usernameprefix}/
        (Later we'll allow generic regexps?)
        
Parameters:
        username, usernameprefix, matching_type

Returns:
        0: success
        <>0: no match
******************************************************************************/
static int match_username(
        const char *    username, 
        const char *    regex,
        unsigned short  matching_type)
{
    int rc = -999;
    int i;
    const char *  rest = NULL;
    size_t regexlen;

    /* Protect against NULL strings and check for empty regexp or too small
     * username */
    if (username==NULL || regex==NULL ||
	(regexlen=strlen(regex))==0 || strlen(username)<regexlen)
	return rc;

    /* We know strlen(username)>=strlen(regexlen) */
    rc = strncmp(regex, username, regexlen);
    if (rc != 0)
        return rc;
    /*
     * Check if the string following the prefix consists entirely of digits
     * and that it is non-zero
     */
    if ((matching_type&MATCH_STRICT_PREFIX_NUM) == MATCH_STRICT_PREFIX_NUM)
    {
	rest=username+regexlen;
	/* Rest can be empty if username == regex, but that's not allowed here */
	if (rest[0]=='\0')
	    return -999;

	/* Check rest is digit only */
	for (i=0; rest[i]; i++)	{
	    if (!isdigit(rest[i]))
		return 1;
	}
    }

    return 0;
}


/******************************************************************************
                       Start of gridmapdir functions

 These all use the environment variable MAPDIR
  (a) if not set, then the gridmapdir functions are not used
  (b) the value it is set to defines the gridmap directory
      (eg export MAPDIR=/etc/grid-security/gridmapdir)

******************************************************************************/

#include <utime.h>
#include <errno.h>
#include <ctype.h>
#include <dirent.h>
#include <unistd.h>
#include <sys/stat.h>
#include <pwd.h>
#include <sys/types.h>

/******************************************************************************
Function:   gridmapdir_otherlink
Description:
        find another link in MAPDIR to the same inode as firstlink
        and change the modification time of firstlink to now (so that we
        always know when this pair was last used)
        
Parameters:
        firstlink, the filename of the link we already know

Returns:
        a pointer to the other link's filename (without path) or NULL if none
        found (this is malloc'd and will need freeing)

******************************************************************************/
static char 
*gridmapdir_otherlink(char *   firstlink)
{
     int            ret;
     char           *firstlinkpath, *otherlinkdup, *otherlinkpath,
                    *gridmapdir;
     struct dirent  *gridmapdirentry;
     DIR            *gridmapdirstream;
     struct stat    statbuf;
     ino_t          firstinode;


     gridmapdir = getenv("MAPDIR");
     if (gridmapdir == NULL) return NULL;
     
     firstlinkpath = malloc(strlen(gridmapdir) + 2 + strlen(firstlink));
     sprintf(firstlinkpath, "%s/%s", gridmapdir, firstlink);     
     ret = stat(firstlinkpath, &statbuf);
     free(firstlinkpath);   
     if (ret != 0) return NULL;
     if (statbuf.st_nlink == 1) return NULL;
     if (statbuf.st_nlink > 2)
     {
         fprintf(stderr,"lcmaps_gridlist(): too many hardlinks found (%ld) for %s\n", (long int)statbuf.st_nlink, firstlink);
         return NULL;
     }
     
     firstinode = statbuf.st_ino; /* save for comparisons */
          
//     gridmapdirstream = opendir(gridmapdir);
     if ((gridmapdirstream = opendir(gridmapdir)) == NULL)
     {
         fprintf(stderr,"lcmaps_gridlist(): error opening directory %s: %s\n", gridmapdir, strerror(errno));
         return NULL;
     }

     if (gridmapdirstream != NULL)
     {
         while ((gridmapdirentry = readdir(gridmapdirstream)) != NULL)
         {       
                 if (strcmp(gridmapdirentry->d_name, firstlink) == 0) continue;
           
                 otherlinkpath = malloc(strlen(gridmapdir) + 2 + 
                                        strlen(gridmapdirentry->d_name));
                 sprintf(otherlinkpath, "%s/%s", gridmapdir, 
                                            gridmapdirentry->d_name);

                 ret = stat(otherlinkpath, &statbuf);     
                 if ((ret == 0) && (statbuf.st_ino == firstinode))
                 {
                      utime(otherlinkpath, (struct utimbuf *) NULL);
                      free(otherlinkpath);
                      otherlinkdup = strdup(gridmapdirentry->d_name);
                      closedir(gridmapdirstream);     
                      return otherlinkdup;
                 }
                 else free(otherlinkpath);
         }
         
         closedir(gridmapdirstream);     
     }

     return NULL;
}

/******************************************************************************
Function:   gridmapdir_urlencode
Description:
        Convert string to URL encoded and return pointer to the encoded
        version, obtained through malloc. Calling routine must free
        this. Here "URL encoded" means anything other than an isalnum()
        goes to %HH where HH is its ascii value in hex; also A-Z => a-z 
        This name is suitable for filenames since no / or spaces.
        The part of the poolindex that contains the local group mappings 
        and the mapcounter is copied as-is in the encoded URL. The subject DN
        and this latter part is separated by the magic character '\001', which
        is translated in the encoded string as ':'

Parameters:
        rawstring, the string to be converted

Returns:
        a pointer to the encoded string or NULL if the malloc failed

******************************************************************************/
char 
*gridmapdir_urlencode(char * rawstring)
{
     int          encodedchar = 0, rawchar = 0;
     char *       encodedstring;
     
     encodedstring = (char *) malloc(3 * strlen(rawstring) + 1);
     
     if (encodedstring == NULL) return (char *) NULL;

     while (rawstring[rawchar] != '\0')
     {
           if (isalnum(rawstring[rawchar]))
           {
               encodedstring[encodedchar] = tolower(rawstring[rawchar]);
               ++rawchar;
               ++encodedchar;
           }
           else if (rawstring[rawchar] == '\001' ) {
               sprintf(&encodedstring[encodedchar],":%s",rawstring+rawchar+1);
               encodedchar += strlen(rawstring+rawchar+1) + 1;
               break;
           } 
           else 
           {
               sprintf(&encodedstring[encodedchar], "%%%02x", 
                                               rawstring[rawchar]);
               ++rawchar;
               encodedchar = encodedchar + 3;
           }        
     }

     encodedstring[encodedchar] = '\0';
     
     return encodedstring;
}

/******************************************************************************
Function:   gridmapdir_newlease
Description:
        Search for an unleased local username to give to the globus user
        corresponding to encodedfilename, and then lease it.

Parameters: 
        encodedfilename, URL-encoded globus client name and pathname of 
           the globus user who requested authentication 
        usernameprefix, the prefix of acceptable usernames (or "\0")

Returns:
        int
        0 == good
        1 == noleaseavailable (any more) in gridmapdir
        2 == Failure in opening the gridmapdir
        3 == other failure
******************************************************************************/

/* void */
 /* is linked in in plugin, so should not be exported, because otherwise available to other */
 /* plugins */
static int
gridmapdir_newlease(char *     encodedglobusidp,
                    char *     usernameprefix,
          unsigned short       matching_type)
{
     int            ret;
     char           *userfilename, *encodedfilename, *gridmapdir;
     struct dirent  *gridmapdirentry;
     DIR            *gridmapdirstream;
     struct stat    statbuf;

     gridmapdir = getenv("MAPDIR");
     if (gridmapdir == NULL) return 2;

     encodedfilename = malloc(strlen(gridmapdir) + (size_t) 2 + 
                              strlen(encodedglobusidp));
     sprintf(encodedfilename, "%s/%s", gridmapdir, encodedglobusidp);
     ret = stat(encodedfilename, &statbuf);
     if (ret == 0)
     {
         /* encodedfilename exists, but has only one link (itself) --> remove it */
         if (statbuf.st_nlink == 1)
         {
             lcmaps_log(LOG_INFO, "%s: removing solitary leasename: %s\n", __func__, encodedfilename);
             unlink(encodedfilename);
         }
     }

     /*
      * Previously it was not checked if gridmapdir existed ! Do it now
      */
     if ((gridmapdirstream = opendir(gridmapdir)) == NULL)
     {
         lcmaps_log(LOG_ERR, "%s: error opening directory %s: %s\n", __func__, gridmapdir, strerror(errno));
         return 2;
     }

     while ((gridmapdirentry = readdir(gridmapdirstream)) != NULL)
     {
       /* we dont want any files that dont look like acceptable usernames */
       if ((*(gridmapdirentry->d_name) == '%') || 
           (strcmp(gridmapdirentry->d_name, "root") == 0))   continue;
       else if (*(gridmapdirentry->d_name) == '.')           continue;
       else if (index(gridmapdirentry->d_name, '~') != NULL) continue;
       else if (match_username(gridmapdirentry->d_name, usernameprefix, matching_type) != 0)
            continue;

       userfilename = malloc(strlen(gridmapdir) + (size_t) 2 + 
                             strlen(gridmapdirentry->d_name));
       sprintf(userfilename, "%s/%s", gridmapdir, gridmapdirentry->d_name);
       stat(userfilename, &statbuf);


       if (statbuf.st_nlink == 1) /* this one isnt leased yet */
       {
           ret = link(userfilename, encodedfilename);
           if (ret != 0)
           {
               /* link failed: this is probably because a VERY lucky
                  other process has obtained a lease for encodedfilename 
                  while we were faffing around */
               /* OR the encodedfilename is already linked to another lease name ! */
               lcmaps_log(LOG_ERR, "%s: could not hardlink \"%s\" with \"%s\"\n", __func__, userfilename, encodedfilename);
               free(userfilename);
               closedir(gridmapdirstream);
               free(encodedfilename);
               return 3;
           }
           free(userfilename);

           stat(encodedfilename, &statbuf);
           if (statbuf.st_nlink > 2)
           {
               /* two globusIDs have grabbed the same username: back off */
               unlink(encodedfilename);
               lcmaps_log(LOG_NOTICE, "%s: Two ID have grabbed the same account, backing off. To preserve a clean gridmapdir state: Unlinking \"%s\"\n", __func__, encodedfilename);
               continue;
           }

           closedir(gridmapdirstream);
           free(encodedfilename);
           return 0; /* link worked ok, so return */
       }
       else
       {
           free(userfilename); /* already in use, try next one */
       }
     }

     lcmaps_log(LOG_INFO, "%s: Unable to lease a poolaccount in gridmapdir \"%s\", searching in the pool \"%s\" for \"%s\"\n", __func__, gridmapdir, usernameprefix, encodedglobusidp);

     closedir(gridmapdirstream);
     free(encodedfilename);
     return 1; /* no unleased names left: give up */     
}
     
/******************************************************************************
Function:   gridmapdir_userid
Description:
        This is equivalent to globus_gss_assist_gridmap but for the dynamic
        user ids in the gridmapdir: maps a globusID to a local unix user id,
        either one already leased, or calls gridmapdir_newlease() to obtain 
        a new lease. This is called by globus_gss_assist_gridmap if the 
        local user id in the static gridmap file begins . (for a dynamic id)

Parameters: 
        globusidp, globus client name who requested authentication 
        usernameprefix, prefix of the local usernames which would 
               be acceptable (or "\0" )
        *userid returned userid name for local system. 

Returns:
        0 on success
        !=0 on failure

******************************************************************************/
static int
gridmapdir_userid(char *     globusidp,
                  char *     usernameprefix,
                  char **    useridp,
        unsigned short       matching_type)
{
     char *encodedglobusidp  = NULL;
     int   rc = 0;


     if (getenv("MAPDIR") == NULL) return 1; /* MAPDIR defined? */

     if (globusidp[0] != '/') return 1; /* must be a proper subject DN */

     encodedglobusidp = gridmapdir_urlencode(globusidp);

     /* Check if this lease is already available */
     if (*useridp = gridmapdir_otherlink(encodedglobusidp))
        lcmaps_log_debug (5, "%s: Found an existing lease: %s\n", __func__, *useridp);

     /*
      * Check if we're running in safe mode (ONLY_USE_EXISTING_LEASE flag is set)
      * If so, we will neither create a new lease nor (re)move leases and
      * only return a username if it matches the prefix
      */
     if ((matching_type&ONLY_USE_EXISTING_LEASE) == ONLY_USE_EXISTING_LEASE)
     {
         if (*useridp == NULL) return 1; /* Will not try to create another lease! */
         if ((match_username(*useridp, usernameprefix, matching_type)) != 0)
         {
             free(*useridp);
             *useridp = NULL;
             return 1;
         }
         /* Success: we found an existing lease */
         return 0;
     }

     /*
      * The lease is inconsistent with the usernameprefix from the grid-mapfile.
      * If OVERRIDE_INCONSISTANCY is set, remove the lease and set *useridp to NULL and
      * start over again.
      */
     if ((*useridp != NULL) && ((match_username(*useridp, usernameprefix, matching_type)) != 0))
     {
         char *encodedfilename = NULL;
         char *gridmapdir      = NULL;


         /* full file path building */
         gridmapdir = getenv("MAPDIR");
         encodedfilename = malloc(strlen(gridmapdir) + (size_t) 2 + strlen(encodedglobusidp));
         sprintf(encodedfilename, "%s/%s", gridmapdir, encodedglobusidp);
 
         /* unlink every DN that comes here if the 'OVERRIDE_INCONSISTENCY' parameter is added to 'matching_type' */
         if ((matching_type&OVERRIDE_INCONSISTANCY) == OVERRIDE_INCONSISTANCY)
         {
             struct stat    statbuf;
             /* unlink the encodedglobusid (represents DN) */
             stat(encodedfilename, &statbuf);
             if (statbuf.st_nlink == 2)
             {
                 lcmaps_log (LOG_NOTICE,"%s: (Re)moving hard link named \"%s\" (was linked to %s, will be linked to %s*)\n", __func__, encodedfilename, *useridp, usernameprefix);
                 unlink(encodedfilename);
             }
             else
             {
                 lcmaps_log(LOG_ERR,"%s: Something really wrong, number of hardlinks = %ld (should be 2 for %s and lease %s)\n", __func__, (long int)statbuf.st_nlink, encodedfilename, *useridp);
                 /* Cleaning */
                 if (encodedfilename)
                 {
                     free (encodedfilename);
                     encodedfilename = NULL;
                 }
                 if (*useridp)
                 {
                     free(*useridp);
                     *useridp = NULL;
                 }
                 return 1;
             }
             /* Cleaning */
             if (encodedfilename)
             {
                 free (encodedfilename);
                 encodedfilename = NULL;
             }
             if (*useridp)
             {
                 free(*useridp);
                 *useridp = NULL;
             }
         }
         else
         {
             lcmaps_log (LOG_NOTICE, "%s: hard link named \"%s\" points to the wrong lease: %s\n", __func__, encodedfilename, *useridp);
             /* Cleaning */
             if (encodedfilename)
             {
                 free (encodedfilename);
                 encodedfilename = NULL;
             }
             if (*useridp)
             {
                 free(*useridp);
                 *useridp = NULL;
             }
             return 1;
         }
     }


     /* maybe no lease yet (*useridp == NULL) */
     if (*useridp == NULL) 
     {
         lcmaps_log_debug (5, "%s: trying to create a new lease.\n", __func__);

         /* try making a lease */
         /* 0 == good
            1 == noleaseavailable (any more) in gridmapdir
            2 == Failure in opening the gridmapdir
            3 == other failure */
         if ((rc = gridmapdir_newlease(encodedglobusidp, usernameprefix, matching_type)) == 0)
         {
             /* check lease */ 
             *useridp = gridmapdir_otherlink(encodedglobusidp); 

             /* check if now there is a consistent lease - possibly made by someone else */
             if ((*useridp == NULL) || (match_username(*useridp, usernameprefix, matching_type) != 0)) 
             {
                 free(encodedglobusidp);
                 return 1; /* still no good */
             }
             else
             {
                 /* Success */
                 free(encodedglobusidp);
                 return 0;
             }
         }
         else if (rc == 1)
         {
             lcmaps_log (LOG_ERR, "%s: Error: Failed to select a lease in the gridmapdir in pool \"%s\". Hint: The pool might be unavailable, not setup or full.\n", __func__, usernameprefix);
             free(encodedglobusidp);
             return 1; /* still no good */
         }
         else if (rc == 2)
         {
             lcmaps_log (LOG_ERR, "%s: Error: Failed to open and/or use the gridmapdir. Unresolvable.\n", __func__);
             free(encodedglobusidp);
             return 1; /* still no good */
         }
         else
         {
             lcmaps_log (LOG_ERR, "%s: Error: Odd error in using the gridmapdir. Error code of gridmapdir_newlease() was %d\n", __func__, rc);
             free(encodedglobusidp);
             return 1; /* still no good */
         }
     }

     free(encodedglobusidp);
     return 0;
}

/******************************************************************************
Function:   gridmapdir_globusid
Description:
        This is equivalent to globus_gss_assist_map_local_user but for the 
        dynamic user ids in the gridmapdir: search through leases to find
        which globusID corresponds to a local unix user id.
        This is called by globus_gss_assist_map_local_user 

Parameters: 
        globus client name who requested authentication 
        *userid returned userid name for local system. 

Returns:
        0 on success
        !=0 on failure

******************************************************************************/

static int
gridmapdir_globusid(char *     useridp,
                    char **    globusidp)
{
     int              encodedptr = 0, decodedptr = 0;
     char             *encodedglobusidp; 

     if (useridp[0] == '/') return 1; /* must not be a subject DN */

     encodedglobusidp = gridmapdir_otherlink(useridp);

     if (encodedglobusidp == NULL) return 1; /* not leased */

     *globusidp = malloc(strlen(encodedglobusidp));

     while (encodedglobusidp[encodedptr] != '\0')
     {
            if (encodedglobusidp[encodedptr] == ':')
            {
              sprintf((*globusidp)+decodedptr,"\001%s",encodedglobusidp+encodedptr+1);
              decodedptr += strlen(encodedglobusidp+encodedptr);
              break;
            }
            if (encodedglobusidp[encodedptr] != '%')
            {
                (*globusidp)[decodedptr] = encodedglobusidp[encodedptr];
                ++encodedptr;
                ++decodedptr;
            }
            else /* must be a %HH encoded character */
            {
                /* even paranoids have enemies ... */
                if (encodedglobusidp[encodedptr+1] == '\0') break;
                if (encodedglobusidp[encodedptr+2] == '\0') break;

                (*globusidp)[decodedptr] = 
                   xdigit_to_value(encodedglobusidp[encodedptr+1]) * 16 +
                   xdigit_to_value(encodedglobusidp[encodedptr+2]);

                encodedptr = encodedptr + 3;
                ++decodedptr;
            }
     }

     free(encodedglobusidp);
     (*globusidp)[decodedptr] = '\0';
     return 0;
}

/******************************************************************************
Function:   gridmapdir_userok
Description:
        This is equivalent to globus_gss_assist_userok but for the dynamic
        user ids in the gridmapdir: finds the local unix username leased to 
        a globusID and compare with the username being checked.
        This is called by globus_gss_assist_userok if the local user id in
        the static gridmap file is -  (for a dynamic id)

Parameters: 
        globus client name who requested authentication 
        userid to be checked

Returns:
        0 on success (authorization allowed)
        !=0 on failure or authorization denied
                
******************************************************************************/

static int
gridmapdir_userok(char *     globusidp,
                  char *     userid)
{
     char                    *encodedglobusidp, *leasedname;

     if (globusidp[0] != '/') return 1; /* must be a proper subject DN */

     encodedglobusidp = gridmapdir_urlencode(globusidp);
     leasedname       = gridmapdir_otherlink(encodedglobusidp);
     free(encodedglobusidp);

     if (leasedname == NULL) return 1;

     if (strcmp(userid, leasedname) == 0)
     {
         free(leasedname);
         return 0;
     }
     else
     {
         free(leasedname);
         return 1;
     }
}

/******************************************************************************
                     End of gridmapdir functions
******************************************************************************/

/******************************************************************************
Function:   lcmaps_gridlist (formerly known as globus_gss_assist_gridmap.c)
Description:
    EDG: Extended for use with other files than gridmapfiles
    EDG: e.g. vo/group/poolaccount mapfiles
    EDG: if globusidfile==NULL the gridmapfile should
    EDG: be used.

    Routines callable from globus based code to 
    map a globusID to a local unix user

    GRIDMAP environment variable pointing at the
    map file. Defaults to ~/.gridmap 

    A gridmap file is required if being run as root. 

    if being run as a user,it is not required, and defaults to 
    the current user who is running the command. 

    This is the same file used by the gssapi_cleartext
    but will be used with other gssapi implementations which 
    do not use the gridmap file. 

Parameters:
    globusidp:     client name who requested authentication 
    userid:        returned userid name for local system. 
    globusidfile:  name of the file containing globusids
    matching_type: flag that indicates the type of DN matching: 
                   MATCH_INCLUDE, MATCH_EXCLUDE, MATCH_WILD_CHARS,
                   OVERRIDE_INCONSISTANCY, ONLY_USE_EXISTING_LEASE 
    searchstr:     string to be matched (or not), typically "."
    leasename:     name of the lease for gridmapdir/groupmapdir functionality


Returns:

    LCMAPS_MOD_SUCCESS: on succes
    LCMAPS_MOD_ENTRY:   entry found
    LCMAPS_MOD_NOENTRY: no entry found
    LCMAPS_MOD_NOFILE:  globusidfile not found
******************************************************************************/
int
lcmaps_gridlist(
    char *          globusidp,
    char **         useridp,
    char *          globusidfile,
    unsigned short  matching_type,
    char *          searchstr,
    const char *    leasename
) 
{
    gridmap_line_t *                    gline = NULL;
    int                                 rc;
    char                                *usernameprefix;
    int                                 ret;
    char *                              newglobusidp = NULL;

    /* Check arguments */
    if (globusidp == NULL)
    {
        lcmaps_log(LOG_ERR,"%s: No Subject-DN (and/or extra credentials) provided. to be matched\n", __func__);
        return LCMAPS_MOD_NOENTRY;
    }

    lcmaps_log_debug(4,"lcmaps_gridlist: called/ globusidp=%s,useridp=%08x,gidfile=%08x\n",
                       globusidp,useridp,globusidfile);

    *useridp = NULL;


    rc = gridmap_find_dn(globusidp, &gline, globusidfile, matching_type, searchstr);
    lcmaps_log_debug(4,"lcmaps_gridlist: gridmap_find_dn rc=%d (for client %s)\n", rc,globusidp);
    if (rc == -1)
    {
        struct passwd *                 pw;

        /*
         * We failed to open the gridmap file.
         * If we are not running as root, then return our own username.
         * This allows someone to run a daemon as themselves without
         * having to have a gridmap file.
         */

        /* EDG: This will change as soon as the LCMAPS will run under its */
        /* EDG: own uid                                                 */
        if ((geteuid() == 0) ||
            (getuid() == 0)) {
            lcmaps_log(LOG_ERR,"lcmaps_gridlist(): failure, no gridmapfile\n");
            if (*useridp) 
            {
                free(*useridp);
                *useridp = NULL;
            }
            return LCMAPS_MOD_NOFILE;
        }

        pw = getpwuid(getuid());

        if ((pw == NULL) ||
            (pw->pw_name == NULL)) {
            lcmaps_log(LOG_ERR,"lcmaps_gridlist(): failure, no gridmapfile and calling user id cannot be determined.\n");
            if (*useridp) 
            {
                free(*useridp);
                *useridp = NULL;
            }
            return LCMAPS_MOD_NOFILE;
        }

        *useridp = strdup(pw->pw_name);
        return  LCMAPS_MOD_SUCCESS;
    }
    else if (rc == 1)
    {
        /*
         * EDG: We are NOT using the gridmapfile and cannot open the globusidfile:
         * EDG: failure
         */
        lcmaps_log(LOG_ERR,"lcmaps_gridlist(): failure, no mapfile. Tried to use: \"%s\"\n", globusidfile);
        if (*useridp) 
        {
            free(*useridp);
            *useridp = NULL;
        }
        return LCMAPS_MOD_NOFILE;
    }


    if (gline != NULL)
    {
        /* Check if we need only the dn in the list (LCAS usage),
         * or we have to do something with the mapping identity (uid/gid,poolid ...)
         * (LCMAPS usage)
         */
        if ((matching_type&MATCH_ONLY_DN) == MATCH_ONLY_DN)
        {
            /* Found an entry and that's enough */
            lcmaps_log_debug(4,"lcmaps_gridlist(): We have a gline (dn entry) and we don't check any further\n");
            gridmap_free_gridmap_line(gline);
            gline = NULL;
            return LCMAPS_MOD_ENTRY;
        }
        lcmaps_log_debug(4,"lcmaps_gridlist(): We have a gline.\n");
        if ((gline->user_ids == NULL) ||
            (gline->user_ids[0] == NULL))
        {
            /*
             * If we get here then something in this code is broken
             * or the gridmap file is badly formatted or, most likely,
             * both.
             */
            lcmaps_log(LOG_ERR,"lcmaps_gridlist(): We have a malformatted gline. Return failure.\n");
            gridmap_free_gridmap_line(gline);
            gline = NULL;
            return LCMAPS_MOD_FAIL;
        }

        /* First user id is default */
        *useridp = strdup(gline->user_ids[0]);
        if (*useridp == NULL)
        {
            /* strdup() failed */
            lcmaps_log(LOG_ERR,"lcmaps_gridlist(): strdup(gline->user_ids[0]) failed\n");
            return LCMAPS_MOD_FAIL;
        }
        lcmaps_log_debug(4,"lcmaps_gridlist(): userid found: %s\n",*useridp);

        gridmap_free_gridmap_line(gline);
        gline = NULL;

        /* need to use gridmapdir, usually searchstr == "." */
        if (searchstr)
        {
            if ( ((matching_type&MATCH_INCLUDE) == MATCH_INCLUDE) &&
                 (strncmp((*useridp), searchstr, strlen(searchstr)) == 0) )
            {
                usernameprefix = strdup(&((*useridp)[strlen(searchstr)]));
                if (*useridp)
                {
                    free(*useridp);
                    *useridp = NULL;
                }
                lcmaps_log_debug(4,"lcmaps_gridlist(): usernameprefix: %s\n",usernameprefix);
                /* check if an alternative name for the lease should be used.             */
                /* this was introduced to create the lease names of the VOMS poolaccounts */
                if (leasename != NULL)
                {
                    newglobusidp = strdup(leasename);
                }
                else
                {
                    newglobusidp = strdup(globusidp);
                }
                lcmaps_log_debug(4,"lcmaps_gridlist(): newglobusidp: %s\n",newglobusidp);
                ret = gridmapdir_userid(newglobusidp, usernameprefix, useridp, matching_type);

                if (*useridp)
                    lcmaps_log_debug(4,"lcmaps_gridlist(): final userid found: %s\n",*useridp);

                if (usernameprefix)
                {
                    free(usernameprefix);
                    usernameprefix = NULL;
                }
                if (newglobusidp)
                {
                    free(newglobusidp);
                    newglobusidp = NULL;
                }
                if (ret == 0)
                {
                    return LCMAPS_MOD_SUCCESS;
                }
                else
                {
                    return LCMAPS_MOD_FAIL;
                }
            }
        }
    }
    else
    {
        /* No entry found in gridmap file for this user */
        lcmaps_log_debug(1,"lcmaps_gridlist(): no entry found for %s\n", globusidp);
        return LCMAPS_MOD_NOENTRY;
    }

    /* Success */
    return LCMAPS_MOD_SUCCESS;

} /* lcmaps_gridlist() */



/******************************************************************************
                           Internal Functions
******************************************************************************/

/******************************************************************************
Function:   gridmap_default_path
Description:
    Determine and return the path to the gridmap file.

Parameters:
    ppath, a pointer to a pointer that will be set to an allocated
    string.

Returns:
    0 on success, non-zero on error.

******************************************************************************/

static
int
gridmap_default_path(char **        ppath)
{
    char                gridmap[256];


    /* the following logic is taken from the gssapi_cleartext
     * globusfile.c. Since it needs this same information,
     * but other gssapi's may not, we duplicate the parsing
     * of the gridmap file. 
     */
    if (getuid() == 0)
    {
        char *char_p;

        if ( ((char_p = (char*) getenv("GRIDMAP")) != NULL) ||
             ((char_p = (char*) getenv("GLOBUSMAP")) != NULL) ||
             ((char_p = (char*) getenv("globusmap")) != NULL) ||
             ((char_p = (char*) getenv("GlobusMap")) != NULL) ) {

            strncpy(gridmap, char_p, sizeof(gridmap)) ;

        } else
            strcpy(gridmap, "/etc/grid-security/grid-mapfile") ;

    }
    else
    {
        char *char_p;

        if ( ((char_p = (char*) getenv("GRIDMAP")) != NULL) ||
             ((char_p = (char*) getenv("GLOBUSMAP")) != NULL) ||
             ((char_p = (char*) getenv("globusmap")) != NULL) ||
             ((char_p = (char*) getenv("GlobusMap")) != NULL) ) {

            strncpy(gridmap, char_p, sizeof(gridmap)) ;
        }
        else
        {
            if ( ((char_p = (char*) getenv("home")) != NULL) ||
             ((char_p = (char*) getenv("Home")) != NULL) ||
             ((char_p = (char*) getenv("HOME")) != NULL)) {

                strcpy(gridmap, char_p);
                strcat(gridmap, "/.gridmap");

            } else {
                strcpy(gridmap,".gridmap") ;
            }
        }
    }

    /* Make certain that no buffer overflow occurred */
    if (strlen(gridmap) > sizeof(gridmap))
        return -1;

    *ppath = strdup(gridmap);

    if (ppath == NULL)
        return -1;

    return 0;

} /* gridmap_default_path() */


/******************************************************************************
Function:   gridmap_find_dn
Description:
    Locate the entry for the given DN in the default gridmap file.

Parameters:
    dn, the name to search for.
    globusidfile: name of the file containing globusids
    if globusidfile==NULL the gridmapfile should
    be used.

    gline, a pointer to a pointer that will be set to point at
    the gridmap_line_t structure containing the line information.
    Will be set to NULL if the line is not found.

Returns:
    0 on success,
    -1: Can not find gridmapfile
    1: Can not find globusidfile (other than gridmapfile)

******************************************************************************/

static
int
gridmap_find_dn(const char * const  dn,
                gridmap_line_t **   gline,
                char *          globusidfile,
                unsigned short      matching_type,
                char *              searchstr) 
{
    char *              gridmap_path = NULL;
    char *              open_mode = "r";
    char *              tmp_name;
    FILE *              gmap_stream = NULL;
    int                 found = 0;
    gridmap_line_t *    gline_tmp;
    int i;


    /* Check arguments */
    if (dn == NULL)
        goto failure;

    if (globusidfile==NULL) /* EDG: use gridmapfile */
    {
        if (gridmap_default_path(&gridmap_path) != 0)
            goto failure;
    }
    else
    {
        gridmap_path=strdup(globusidfile);
    }

    gmap_stream = fopen(gridmap_path, open_mode);

    if (gmap_stream == NULL)
    goto failure;

    free(gridmap_path);
    gridmap_path = NULL;

    do
    {
        char                line[1024];


        if (fgets(line, sizeof(line), gmap_stream) == NULL)
            break;      /* EOF or error */

        if (gridmap_parse_line(line, &gline_tmp) != 0)
            continue;       /* Parse error */


        /* Modify this:
         *   make function that compares dn with gline_tmp->dn
         *   that uses wildcharacters 
         */
        if ((gline_tmp != NULL) && (match_dn(dn, gline_tmp->dn, matching_type) == 0))
        {
            /*
             * Pool account of Local account gathering with INCLUSION or EXCLUSION of 
             * the searchstr @ the beginning of the tmp_name
             */

            lcmaps_log_debug(4," Found match in the mapfile.\n");
            /* user_ids might be zero, so test it first */
            /* if MATCH_ONLY_DN flag is set then user_ids is allowed to be zero for a positive result */
            if ((matching_type&MATCH_ONLY_DN) == MATCH_ONLY_DN)
            {
                found = 1;
            }
            else if (gline_tmp->user_ids != NULL)
            {
                if (searchstr)
                {
		    if ((matching_type & MATCH_EXACT) == MATCH_EXACT)	{
			for (i=0; (tmp_name=gline_tmp->user_ids[i])!=NULL; i++)	{
			    if (strcmp(tmp_name, searchstr) == 0)   {
				/* Swap current entry with first, to make match
				 * the first entry */
				gline_tmp->user_ids[i]=gline_tmp->user_ids[0];
				gline_tmp->user_ids[0]=tmp_name;
				found = 1;
				break;
			    }
			}
		    } else {
			tmp_name = (gline_tmp->user_ids[0]);
			if ( ((matching_type&MATCH_INCLUDE) == MATCH_INCLUDE) &&
			     (strncmp(tmp_name, searchstr, strlen(searchstr)) == 0))
			{
			    found = 1;
			}
			else if ( !((matching_type&MATCH_INCLUDE) == MATCH_INCLUDE) &&
				 (strncmp(tmp_name, searchstr, strlen(searchstr)) != 0))
			{
			    found = 1;
			} 
			else
			{
			    gridmap_free_gridmap_line(gline_tmp);
			    gline_tmp = NULL;
			    found = 0;
			}
		    }
                }
                else
                {
                    found = 1;
                }
            }
            else
            {
                gridmap_free_gridmap_line(gline_tmp);
                gline_tmp = NULL;
                found = 0;
            }
        }
        else
        {
            lcmaps_log_debug(4," no match\n");
            gridmap_free_gridmap_line(gline_tmp);
            gline_tmp = NULL;
        }

    } while (!found);

    fclose(gmap_stream);
    gmap_stream = NULL;

    if (found)
    *gline = gline_tmp;
    else
    *gline = NULL;

    return 0;

 failure:

    if (gridmap_path != NULL)
        free(gridmap_path);

    if (gmap_stream)
        fclose(gmap_stream);

    /*
     * Be sure that if the gridmap file is not used and we can not find 
     * globusidfile the return type is more fatal: 1
     */
    if ((gmap_stream == NULL) && (globusidfile!=NULL))
    {
        return 1;
    }
    else
    {
        return -1;
    }

} /* gridmap_find_dn() */

  
/******************************************************************************
Function:   gridmap_parse_line
Description:
    Given a line from the gridmap file, parse it returning
    a gridmap_line_t structure. line is modified during parsing.

    The format of the line is expected to be:

    <DN> <userid>[,<userid>[,<userid>...]]

    Leading and trailing whitespace is ignored.

    userids must only have a comma between them, no whitespace.

    Anything after the userids is ignored.

    Anything after an unescaped comment character is ignored.

Parameters:
    line, a pointer to the line from the file (NUL-terminated string)

    gline, a pointer to a pointer that will be set to point at
    the gridmap_line_t structure containing the line information.
    If the line contains no content, gline will be set to NULL.

Returns:
    0 on success, non-zero on error.

******************************************************************************/

static
int
gridmap_parse_line(char *           line,
             gridmap_line_t **  gline)
{
    char *              dn_end;
    char *              parsed_dn = NULL;
    char **             userids = NULL;
    int                 num_userids = 0;
    int                 userid_slots = 0;
    gridmap_line_t *    gline_tmp = NULL;

    /* Check arguments */
    if ((line == NULL) || (gline == NULL))
        goto error;

    /* Skip over leading whitespace */
    line += strspn(line, WHITESPACE_CHARS);

    /* Check for comment at start of line and ignore line if present */
    if (strchr(COMMENT_CHARS, *line) != NULL) 
    {
        /* Ignore line, return NULL gline */
        *gline = NULL;
        return 0;
    }

    /* Check for empty line */
    if (*line == NUL)
    {
        /* Empty line, return NULL gline. */
        *gline = NULL;
        return 0;
    }

    /* Is DN quoted? */
    if (strchr(QUOTING_CHARS, *line) != NULL)
    {
        /*
         * Yes, skip over opening quote and look for unescaped
         * closing double quote
         */
        line++;
        dn_end = line;

        do
        {
            dn_end += strcspn(dn_end, QUOTING_CHARS);

            if (*dn_end == NUL)
                goto error; /* Missing closing quote */

        /* Make sure it's not escaped */
        }
        while (strchr(ESCAPING_CHARS, *(dn_end - 1)) != NULL);
    }
    else
    {
        /* No, just find next whitespace */
        dn_end = line + strcspn(line, WHITESPACE_CHARS);

        if (*dn_end == NUL)
            goto error; /* Nothing after DN */
    }

    /* NUL terminate DN and parse */
    *dn_end = NUL;

    if (gridmap_parse_globusid(line, &parsed_dn) != 0)
        return -1;

    /* Skip over closing delim and any whitespace after DN */
    line = dn_end + 1;
    line += strspn(line, WHITESPACE_CHARS);

    /* Parse list of unix user ID seperated by USERID_SEP_CHARS */
    while ((*line != NUL) &&
       (strchr(WHITESPACE_CHARS, *line) == NULL))
    {
        int                 userid_len;

        /* Find end of this userid */
        userid_len = strcspn(line, USERID_TERMINATOR_CHARS);

        /* Make sure we have a slot and if not allocate it */
        if ((num_userids + 1 /* new entry */+ 1 /* for NULL */) > userid_slots)
        {
            char **userids_tmp;


            userid_slots += USERID_CHUNK_SIZE;

            userids_tmp = realloc(userids, userid_slots * sizeof(char *));

            if (userids_tmp == NULL)
                goto error;

            userids = userids_tmp;
        }

        userids[num_userids] = malloc(userid_len + 1 /* for NUL */);

        if (userids[num_userids] == NULL)
            goto error;

        strncpy(userids[num_userids], line, userid_len);
        userids[num_userids][userid_len] = NUL;

        num_userids++;
        userids[num_userids] = NULL;

        line += userid_len;

        /* If we're on a seperator character, skip over it */
        if (strchr(USERID_SEP_CHARS, *line) != NULL)
            line++;
    }

    /*
     * There might be more stuff on the line, but we're ignoring
     * it for now.
     */

    /* Ok, build our gridmap_line_t structure */
    gline_tmp = malloc(sizeof(*gline_tmp));

    if (gline_tmp == NULL)
        goto error;

    gline_tmp->dn = parsed_dn;
    gline_tmp->user_ids = userids;

    *gline = gline_tmp;

    return 0;

 error:
    if (parsed_dn != NULL)
        free(parsed_dn);

    if (userids != NULL) {
        char **userids_tmp = userids;

        while (*userids_tmp != NULL)
            free(*userids_tmp++);

        free(userids);
    }

    if (gline_tmp != NULL)
        free(gline_tmp);

    return -1;

} /* gridmap_parse_line() */



/******************************************************************************
Function:   gridmap_free_gridmap_line
Description:
    Frees all memory allocated to a gridmap_line_t structure.

Parameters:
    gline, pointer to structure to be freed.

Returns:
    Nothing.
******************************************************************************/
static
void
gridmap_free_gridmap_line(gridmap_line_t *gline)
{
    if (gline != NULL)
    {
        if (gline->dn != NULL)
        {
            free(gline->dn);
            gline->dn = NULL;
        }

        if (gline->user_ids != NULL)
        {
            char **userids_tmp = gline->user_ids;

            while (*userids_tmp != NULL)
                free(*userids_tmp++);

            free(gline->user_ids);
            gline->user_ids = NULL;
        }

        free(gline);
    }

} /* gridmap_free_gridmap_line() */



/******************************************************************************
Function:   gridmap_parse_globusid
Description:
    Given a pointer to a string containing the globusid from the
    gridmap file, return a pointer to a string containing the
    parsed from of the id.

    Specifically handle backslashed characters - e.g. '\\',
    '\x4a' or '\37'.

Parameters:
    unparsed, pointer to unparsed string

    pparsed, pointer to pointer that should be set to point at
    allocated parsed string

Returns:
    0 on success
    non-zero on error.

******************************************************************************/
static
int
gridmap_parse_globusid(
    const char *            unparsed,
    char **             pparsed)
{
    /* Is the current character escaped? (Previous char was backslash) */
    int                 escaped = 0;

    /* Buffer we are putting resulting name into */
    char *              buffer = NULL;

    /* Buffer's length in bytes */
    int                 buffer_len = 0;

    /* And our current position in buffer */
    int                 buffer_index = 0;

    /* Character we're currently looking at */
    char                    unparsed_char;


    /*
     * Check input parameters for legality
     */
    if ((unparsed == NULL) || (pparsed == NULL))
        return -1;

    buffer_len = strlen(unparsed);

    buffer = malloc(buffer_len);

    if (buffer == NULL)
        return -1;

    /*
     * Walk through the name, parsing as we go
     */
    while ((unparsed_char = *(unparsed++)) != NUL)
    {
        /* Unescaped backslash */
        if (strchr(ESCAPING_CHARS, unparsed_char) && !escaped)
        {
            escaped = 1;
            continue;
        }

        /* Escaped hex character - e.g. '\xfe' */
        if ((unparsed_char == 'x') && escaped)
        {
            if (isxdigit(*unparsed) &&
            isxdigit(*(unparsed + 1)))
            {
                /* Set unparsed_char to value represented by hex value */
                unparsed_char =
                    (xdigit_to_value(*unparsed) << 4) +
                    xdigit_to_value(*(unparsed + 1));

                unparsed += 2;
            }
            /* else just fall through */
        }

        /*
         * Ok, we now have the character in unparsed_char to be appended
         * to our output string.
         *
         * First, make sure we have enough room in our output buffer.
         */
        if ((buffer_index + 1 /* for NUL */) >= buffer_len)
        {
            /* Grow buffer */
            char *tmp_buffer;

            buffer_len *= 2;

            tmp_buffer = realloc(buffer, buffer_len);

            if (tmp_buffer == NULL)
            {
                free(buffer);
                return -1;
            }

            buffer = tmp_buffer;
        }

        buffer[buffer_index++] = unparsed_char;
        buffer[buffer_index] = NUL;

        escaped = 0;
    }

    /* XXX What if escaped == 1 here? */

    /* Success */

    *pparsed = buffer;

    return 0;

} /* gridmap_parse_globusid() */



/******************************************************************************
Function:   xdigit_to_value
Description:
    Convert a ascii character representing a hexadecimal digit
    into a integer.

Parameters:
    xdigit, character contain the hex digit.

Returns:
    value contained in xdigit, or -1 on error.

******************************************************************************/

static int
xdigit_to_value(
    char                xdigit)
{
    if ((xdigit >= '0') && (xdigit <= '9'))
        return (xdigit - '0');

    if ((xdigit >= 'a') && (xdigit <= 'f'))
        return (xdigit - 'a' + 0xa);

    if ((xdigit >= 'A') && (xdigit <= 'F'))
        return (xdigit - 'A' + 0xa);

    /* Illegal digit */
    return -1;
} /* xdigit_to_value() */



