/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icontrolscript.h"


#include "ianimator.h"
#include "ianimatorscript.h"
#include "icalculatorparser.h"
#include "icontrolmodule.h"
#include "idatareader.h"
#include "idirectory.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iobjecthelp.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "iviewmodule.h"


//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "icalculatortemplate.h"
#include "iscriptkittemplate.h"

using namespace iParameter;


#define curScript		iRequiredCast<iControlScript>(INFO,self)
#define curRender		curScript->GetAutoRender()
#define curControl		curScript->GetControlModule()
#define curVisual		curScript->GetControlModule()->GetViewModule()


namespace iControlScript_Private
{
	//
	//  Parser that recognizes object keys as variables
	//
	class Parser : public iCalculatorParser
	{

	public:

		Parser() : iCalculatorParser(false)
		{
		}

	protected:

		virtual bool IsVariableLetter(char c) const
		{
			return (this->iCalculatorParser::IsVariableLetter(c) || c==':');
		}
	};


	class Calculator : public iScriptKit::Calculator
	{

	public:

		Calculator(iControlScript *script) : iScriptKit::Calculator(script)
		{
		}

	protected:

		virtual iCalculatorParser* CreateParser() const
		{
			return new Parser;
		}
	};


	template<class T>
	T* QueryKeyData(const iControlScript *script, const iObjectKey *key, int dim)
	{
		T* data = new T[dim];
		if(data != 0) 
		{
			if(script->GetControlModule()->QueryValue(*key,data,dim)) return data; else
			{
				delete [] data;
				return 0;
			}
		}
		else return 0;
	}

	template<class T>
	bool UpdateVariableFromKeyData(iScriptKit::Value *val, T *data, int dim)
	{
		if(val==0 || data==0) return false;
		if(!val->Morph(dim)) return false;

		//
		//  Fill it in
		//
		int i;
		iScriptKit::Calculator::number_t *arr = val->CalculatorValue()->Data();
		for(i=0; i<dim; i++)
		{
			arr[i] = iScriptKit::Constant<T>::ConvertFromNative(data[i]);
		}
		delete [] data;

		return true;
	}


	//
	//  Show/hide a ViewSubject
	//
	bool Show(iScript *self, const iString &arg)
	{
		const iObjectType *tmp = iObjectTypeRegistry::FindType(arg);
		if(tmp != 0)
		{
			curControl->Show(*tmp,true,true);
			return true;
		}
		else
		{
			self->GetErrorStatus()->Set("Invalid object name.");
			return false;
		}
	}


	bool Hide(iScript *self, const iString &arg)
	{
		const iObjectType *tmp = iObjectTypeRegistry::FindType(arg);
		if(tmp != 0)
		{
			curControl->Show(*tmp,false,curRender);
			return true;
		}
		else
		{
			self->GetErrorStatus()->Set("Invalid object name.");
			return false;
		}
	}


	//
	//  Create/delete an object
	//
	bool Create(iScript *self, const iString &arg)
	{
		const iObjectType *tmp = iObjectTypeRegistry::FindType(arg.Section(" ",0,0));
		if(tmp != 0)
		{
			int vmtype = ModuleType::New;

			if(*tmp == iViewModule::Type())
			{
				iString v = arg.Section(" ",1);
				v.ReduceWhiteSpace();
				if(v == "/copy") vmtype = ModuleType::Copy;
				if(v == "/clone") vmtype = ModuleType::Clone;
			}

			if(!curControl->CreateObject(*tmp,curRender,-1,vmtype))
			{
				self->GetErrorStatus()->Set("Object cannot be created - perhaps, there is not enough memory.");
				return false;
			}
			else return true;
		}
		else
		{
			self->GetErrorStatus()->Set("Invalid object name.");
			return false;
		}
	}


	bool Remove(iScript *self, const iString &arg)
	{
		const iObjectType *tmp = iObjectTypeRegistry::FindType(arg);
		if(tmp != 0)
		{
			if(!curControl->DeleteObject(*tmp,curRender))
			{
				self->GetErrorStatus()->Set("It is not allowed to delete this object.");
				return false;
			}
			else return true;
		}
		else
		{
			self->GetErrorStatus()->Set("Invalid object name.");
			return false;
		}
	}


	//
	//  Save/restore state
	//
	bool SaveState(iScript *self, const iString &arg)
	{
		iString filename(arg);
		if(filename.IsEmpty())
		{
			filename = curControl->GetShell()->GetEnvironment(Environment::Base) + "ifrit.ini";
		}
		else if(filename[0] == '+')
		{
			filename.Replace(0,1,curControl->GetShell()->GetEnvironment(Environment::Base));
		}

		if(!curControl->SaveStateToFile(filename))
		{
			self->GetErrorStatus()->Set("Saving state to a file failed.");
			return false;
		}
		else return true;
	}


	bool LoadState(iScript *self, const iString &arg)
	{
		iString filename(arg);
		if(filename.IsEmpty())
		{
			filename = curControl->GetShell()->GetEnvironment(Environment::Base) + "ifrit.ini";
		}
		else if(filename[0] == '+')
		{
			filename.Replace(0,1,curControl->GetShell()->GetEnvironment(Environment::Base));
		}

		if(!curControl->LoadStateFromFile(filename))
		{
			self->GetErrorStatus()->Set("Loading state from a file failed.");
			return false;
		}
		else return true;
	}


	//
	//  Make a window current
	//
	bool Current(iScript *self, short, int val)
	{
		if(!curControl->SetCurrentViewModuleIndex(val-1))
		{
			self->GetErrorStatus()->Set("Incorrect ViewModule index.");
			return false;
		}
		else return true;
	}


	//
	//  Set default render mode
	//
	bool Render(iScript *self, short, bool val)
	{
		curScript->SetAutoRender(val);
		return true;
	}
};


using namespace iControlScript_Private;


iControlScript* iControlScript::New(iControlModule *cm, iScript *parent)
{
	IERROR_ASSERT(cm);
	return iShellFactory::CreateControlScript(cm,parent);
}


iControlScript::iControlScript(iControlModule *cm, iScript *parent) : iBasicScript(parent), mControlModule(cm)
{
	//
	//  Alias words (shortcuts)
	//
	this->CreateAliasWord(".e","exec"); 
	this->CreateAliasWord(".s","show"); 
	this->CreateAliasWord(".q","hide"); 
	this->CreateAliasWord(".h","help"); 
	this->CreateAliasWord(".lo","list objects"); 
	this->CreateAliasWord(".lp","list properties"); 
	this->CreateAliasWord(".ho","help object"); 
	this->CreateAliasWord(".hp","help property"); 
	this->CreateAliasWord(".p","print"); 
	this->CreateAliasWord(".c","create"); 
	this->CreateAliasWord(".d","delete"); 
	this->CreateAliasWord(".r","render"); 
	this->CreateAliasWord(".u","current-window"); 
	this->CreateAliasWord(".a","animate"); 
	this->CreateAliasWord(".ss","save-state"); 
	this->CreateAliasWord(".ls","load-state"); 
	this->CreateAliasWord(".eo","exec-all-objects"); 
	this->CreateAliasWord(".ew","exec-all-windows"); 

#ifdef ISCRIPT_BACKWARD_COMPATIBLE
	//
	//  Backward compatibility
	//
	this->CreateAliasWord("Vector","VectorField"); 
	this->CreateAliasWord("Tensor","TensorField"); 
#endif

	//
	//  Parameter words
	//
	this->AddConstant(new iScriptKit::Constant<bool>(this,"off",false));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"on",true));

	//
	//  Command words
	//

	//
	//  Statements
	//
	this->AddPrototype< iScriptKit::FunctionCallStatement >("exec",Exec);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("show",Show);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("hide",Hide);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("print",Print);  // prints anything
	this->AddPrototype< iScriptKit::FunctionCallStatement >("create",Create);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("delete",Remove);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("animate",Animate);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("exec-all-objects",ExecAllObjects);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("exec-all-windows",ExecAllWindows);

	this->AddPrototype< iScriptKit::FunctionCallStatement >("save-state",SaveState);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("load-state",LoadState);

	this->AddPrototype< iScriptKit::FunctionCallStatement >("embed animator-script",EmbedAnimatorScript);

	//
	//  Help statements
	//
	this->AddPrototype< iScriptKit::FunctionCallStatement >("help",Help);

	this->AddPrototype< iScriptKit::SimpleStatement >("list objects",ListObjects);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("list properties",ListProperties);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("help object",HelpObject);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("help property",HelpProperty);

	//
	//  Genuine assignment operations to pre-defined variables
	//
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("render",Render);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("current-window",Current);

	mAutoRender = true;
}


iControlScript::~iControlScript()
{
}


iScriptKit::Calculator* iControlScript::CreateCalculatorBody()
{
	return new Calculator(this);
}


bool iControlScript::SatisfyRequestForVariables(const iString &list)
{
	iString w;
	const iObjectKey *key;
	int i, n = list.Contains(',') + 1;
	for(i=0; i<n; i++)
	{
		w = list.Section(",",i,i);
		if(w.Find(iObjectKey::PrefixSeparator()) > -1)
		{
			//
			//  Could be a key
			//
			key = iObjectKeyRegistry::FindKey(w,true);
			if(key != 0)
			{
				if(this->FindVariable(key->PrefixedFullName()) == 0)
				{
					switch(key->Argument())
					{
					case iObjectKey::_Int:
					case iObjectKey::_OffsetInt:
						{
							this->AddVariable(new iScriptKit::Constant<int>(this,key->PrefixedFullName(),0,0));
							this->AddVariable(new iScriptKit::Constant<int>(this,key->PrefixedShortName(),0,0));
							break;
						}
					case iObjectKey::_Bool:
						{
							this->AddVariable(new iScriptKit::Constant<bool>(this,key->PrefixedFullName(),0,0));
							this->AddVariable(new iScriptKit::Constant<bool>(this,key->PrefixedShortName(),0,0));
							break;
						}
					case iObjectKey::_Float:
						{
							this->AddVariable(new iScriptKit::Constant<float>(this,key->PrefixedFullName(),0,0));
							this->AddVariable(new iScriptKit::Constant<float>(this,key->PrefixedShortName(),0,0));
							break;
						}
					case iObjectKey::_Double:
						{
							this->AddVariable(new iScriptKit::Constant<double>(this,key->PrefixedFullName(),0,0));
							this->AddVariable(new iScriptKit::Constant<double>(this,key->PrefixedShortName(),0,0));
							break;
						}
					case iObjectKey::_Color:
						{
							this->AddVariable(new iScriptKit::Constant<int>(this,key->PrefixedFullName(),0,0));
							this->AddVariable(new iScriptKit::Constant<int>(this,key->PrefixedShortName(),0,0));
							break;
						}
					default:
						{
							this->GetErrorStatus()->Set("The object property "+key->PrefixedFullName()+" is not allowed in expressions");
							return false;
						}
					}
				}
				this->UpdateObjectKey(key);
			}
		}
	}

	return true;
}


void iControlScript::UpdateObjectKey(const iObjectKey *key) const
{
	iScriptKit::Value *val;
	
	val = this->FindVariable(key->PrefixedFullName());
	if(val != 0) this->UpdateObjectKeyVariable(key,val);
	
	val = this->FindVariable(key->PrefixedShortName());
	if(val != 0) this->UpdateObjectKeyVariable(key,val);
}


void iControlScript::UpdateObjectKeyVariable(const iObjectKey *key, iScriptKit::Value *val) const
{
	int dim = key->Dimension();
	if(dim <= 0)
	{
		iString s;
		if(this->GetControlModule()->QueryValueAsString(*key,s))
		{
			int n = s.Contains(iObjectKey::FieldSeparator()) - 1;
			if(dim<0 || n<1)
			{
				dim = 1;
			}
			else
			{
				//
				//  Arrays w/o bound checking
				//
				dim = n;
			}
		}
		else
		{
			//
			//  This key is missing - that can only be if it is an empty array
			//
			return;
		}
	}

	bool ok = false;
	switch(key->Argument())
	{
	case iObjectKey::_Int:
	case iObjectKey::_OffsetInt:
		{
			int *data = QueryKeyData<int>(this,key,dim);
			ok = UpdateVariableFromKeyData(val,data,dim);
			break;
		}
	case iObjectKey::_Bool:
		{
			bool *data = QueryKeyData<bool>(this,key,dim);
			ok = UpdateVariableFromKeyData(val,data,dim);
			break;
		}
	case iObjectKey::_Float:
		{
			float *data = QueryKeyData<float>(this,key,dim);
			ok = UpdateVariableFromKeyData(val,data,dim);
			break;
		}
	case iObjectKey::_Double:
		{
			double *data = QueryKeyData<double>(this,key,dim);
			ok = UpdateVariableFromKeyData(val,data,dim);
			break;
		}
	case iObjectKey::_Color:
		{
			int *idata = 0;
			iColor *data = QueryKeyData<iColor>(this,key,dim);
			if(data != 0)
			{
				//
				//  Transform to ints
				//
				idata = new int[dim];
				if(idata != 0)
				{
					int i;
					for(i=0; i<dim; i++) idata[i] = data[i].Red() + 256*(data[i].Green()+256*data[i].Blue());
				}
				delete [] data;
			}
			ok = UpdateVariableFromKeyData(val,idata,dim);
			break;
		}
	}

	if(!ok)
	{
		this->GetErrorStatus()->Set("Unable to update the object property; this is a bug.");
	}
}


//
//  Actually do Exec 
//
bool iControlScript::ExecBody(iScript *self, const iString &request, int mode)
{
	const iObjectKey *key = iObjectKeyRegistry::FindKey(request.Section(iObjectKey::FieldSeparator(),0,0),true);
	if(key == 0)
	{
		self->GetErrorStatus()->Set("Request does not start with a valid object property.");
		return false;
	}

	//
	//  Mini-parser: replaces expressions in between of special symbols {} with their values.
	//
	iString w, ex(request);
	int ic, io;
	//
	//  Remove trailing white space
	//
	io = ex.Length() - 1;
	while(ex.IsWhiteSpace(io)) io--;
	ex = ex.Part(0,io+1);

	while((ic=ex.Find('}')) >= 0)
	{
		w = ex.Part(0,ic+1);
		io = w.Find('{');
		if(io == -1)
		{
			self->GetErrorStatus()->Set("No opening { in front of the closing } found.",0);
			return false;
		}

		w = ex.Part(io+1,ic-io-1);
		if(!self->GetCalculator()->Evaluate(w))
		{
			self->GetErrorStatus()->Set(self->GetCalculator()->GetErrorMessage());
			return false;
		}
		if(self->GetCalculator()->GetResultSize() != 1)
		{
			self->GetErrorStatus()->Set("Only scalar (1-dimensional) expressions are allowed inside object properties.");
			return false;
		}
		ex.Replace(io,ic-io+1,iString::FromNumber(self->GetCalculator()->GetResultData()[0]));
	}

	//
	//  Execute the command
	//
	self->GetErrorStatus()->Monitor(curControl->GetErrorStatus());
	if(!curControl->Execute(ex,curRender,mode) && curControl->GetErrorStatus()->NoError())
	{
		self->GetErrorStatus()->Set("Incorrect command syntax or invalid parameter.");
	}

	if(self->GetErrorStatus()->NoError())
	{
		curScript->UpdateObjectKey(key);
		return true;
	}
	else return false;
}


//
//  Exec a ControlModule command
//
bool iControlScript::Exec(iScript *self, const iString &request)
{
	return ExecBody(self,request,ObjectOption::One|ModuleOption::One|RenderOption::Auto);
}


bool iControlScript::ExecAllObjects(iScript *self, const iString &request)
{
	return ExecBody(self,request,ObjectOption::All|ModuleOption::One|RenderOption::Auto);
}


bool iControlScript::ExecAllWindows(iScript *self, const iString &request)
{
	return ExecBody(self,request,ObjectOption::All|ModuleOption::All|RenderOption::Auto);
}


//
//  Print anything
//
bool iControlScript::Print(iScript *self, const iString &arg)
{
	if(self->IsEmbedded()) return true;

	int i;
	iString w(arg);
	w.ReduceWhiteSpace();
	if(w == "/all")
	{
		iArray<const iScriptKit::Value*> list;
		self->QueryVariables(list);
		for(i=0; i<list.Size(); i++)
		{
			curScript->OutputText("<b>"+list[i]->Name()+"</b>{"+list[i]->GetTypeAsText()+"}: "+list[i]->GetValueAsText());
		}
		return true;
	}

	int io = 0, ic = -1;
	while(io < arg.Length())
	{
		ic = arg.Find(',',ic+1); if(ic == -1) ic = arg.Length();
		w = arg.Part(io,ic-io);
		if(w.Contains('(') != w.Contains(')'))  //  inside parentheses
		{
			if(ic < arg.Length()) continue; else
			{
				curScript->OutputText("Unmatched parenthesis.");
				return false;
			}
		}
		
		w.RemoveWhiteSpace();

		//
		//  Is it a complete type?
		//
		if(w.EndsWith(iObjectKey::PrefixSeparator()))
		{
			w = w.Part(0,w.Length()-iObjectKey::PrefixSeparator().Length());
			const iObjectType *type = iObjectTypeRegistry::FindType(w);
			if(type != 0)
			{
				int i, lmax = 0;
				iString s, s1, out;
				const iObjectKey *ptr;

				iObjectKeyRegistry::InitTraversal(type);
				while((ptr=iObjectKeyRegistry::GetNextKey()) != 0) if(ptr->UnprefixedFullName().Part(0,9) != "___old___")
				{
					if(lmax < ptr->UnprefixedFullName().Length()) lmax = ptr->UnprefixedFullName().Length();
				}

				for(i=0; i<lmax; i++) s += "$";

				out  = "<b>" + type->FullName() + "</b>:<br><hr>";

				iObjectKeyRegistry::InitTraversal(type);
				while((ptr=iObjectKeyRegistry::GetNextKey()) != 0) if(ptr->UnprefixedFullName().Part(0,9) != "___old___")
				{
					s1.Clear();
					curControl->QueryValueAsString(*ptr,s1);
					if(!s1.IsEmpty()) out += "<b>" + ptr->UnprefixedFullName() + "</b>" + s.Part(0,lmax-ptr->UnprefixedFullName().Length()) + " : $$" + s1 + "<br>";
				}
				out.Replace("$","&nbsp;");
				curScript->OutputText(out);
				io = ic + 1;
				continue;
			}
		}

		//
		//  Is it a single property?
		//
		const iObjectKey *key = iObjectKeyRegistry::FindKey(w);
		if(key != 0)
		{
			iString s1, out = "<b>" + key->PrefixedFullName() + "</b>" + " : ";
			curControl->QueryValueAsString(*key,s1);
			out += s1 + "<br>";
			curScript->OutputText(out);
			io = ic + 1;
			continue;
		}


		//
		//  Is it a variable?
		//
		iScriptKit::Value *val = curScript->FindVariable(w);
		if(val != 0)
		{
			curScript->OutputText("<b>"+val->Name()+"</b>{"+val->GetTypeAsText()+"} : "+val->GetValueAsText());
			io = ic + 1;
			continue;
		}


		//
		//  Is it an expression?
		//
		if(self->GetCalculator()->Compile(w))
		{
			if(self->GetCalculator()->Link() && self->GetCalculator()->Execute())
			{
				if(self->GetCalculator()->GetResult()->Use() == 1)
				{
					iScriptKit::Constant<bool> c(self,self->GetCalculator()->GetResult());
					curScript->OutputText("{"+c.GetTypeAsText()+"}: "+c.GetValueAsText());
				}
				else
				{
					iScriptKit::Constant<Calculator::number_t> c(self,self->GetCalculator()->GetResult());
					curScript->OutputText("{"+c.GetTypeAsText()+"} : "+c.GetValueAsText());
				}
			}
			else
			{
				curScript->OutputText(self->GetCalculator()->GetErrorMessage());
				return false;
			}
			io = ic + 1;
			continue;
		}

		//
		//  Could not make sense of it, just print it.
		//
		curScript->OutputText(w);
		io = ic + 1;
	}

	return true;
}


bool iControlScript::Animate(iScript *self, const iString &arg)
{
	static const iString all("/all");
	static const iString clones("/clones");

	if(self->IsEmbedded()) return true;

	if(!curVisual->GetReader()->IsFileAnimatable())
	{
		self->GetErrorStatus()->Set("Please, load an animatable file for the Animator to work.");
		return false;
	}

	if(!arg.IsEmpty() && arg!=all && arg!=clones)
	{
		iString fname(arg);

		iDirectory::ExpandFileName(fname,curControl->GetShell()->GetEnvironment(Environment::Base));

		iFile f(fname);
		if(!f.Open(iFile::_Read,iFile::_Text))
		{
			self->GetErrorStatus()->Set("Animation script file is not found.");
			return false;
		}

		iString st, tmp;
		while(f.ReadLine(tmp)) st += tmp + "\n";
		if(st.IsEmpty())
		{
			self->GetErrorStatus()->Set("Animation script file is empty or unreadable.");
			return false;
		}

		f.Close();

		curVisual->GetAnimator()->GetScript()->SetText(st);
	}

	if(arg == all)
	{
		int i, n = curControl->GetNumberOfViewModules();
		for(i=0; i<n; i++)
		{
			curVisual->GetAnimator()->AddFollower(curControl->GetViewModule(i)->GetAnimator());
		}
	}

	if(arg == clones)
	{
		int i, n = curControl->GetNumberOfViewModules();
		for(i=0; i<n; i++) if(curControl->GetViewModule(i)->IsClone(curVisual))
		{
			curVisual->GetAnimator()->AddFollower(curControl->GetViewModule(i)->GetAnimator());
		}
	}

	self->GetErrorStatus()->Monitor(curVisual->GetAnimator()->GetErrorStatus());
	curVisual->GetAnimator()->Animate();
	curVisual->GetAnimator()->RemoveAllFollowers();

	return true;
}


bool iControlScript::EmbedAnimatorScript(iScript *self, const iString &arg)
{
	if(self->IsEmbedded()) return true;

	if(!curVisual->GetReader()->IsFileAnimatable())
	{
		self->GetErrorStatus()->Set("Please, load an animatable file for the Animator to work.");
		return false;
	}

	iString v(arg);

	//
	//  Replace %% with \n
	//
	v.Replace("%%","\n");

	self->GetErrorStatus()->Monitor(curVisual->GetAnimator()->GetErrorStatus());
	curVisual->GetAnimator()->GetScript()->SetText(v);
	curScript->mAllowChildAccess = true;
	curVisual->GetAnimator()->Animate();
	curScript->mAllowChildAccess = false;

	return true;
}


//
//  Help functions
//
bool iControlScript::Help(iScript *self, const iString &arg)
{
	if(self->IsEmbedded()) return true;

	iString w(arg);
	w.RemoveWhiteSpace();
	if(w == "exec")
	{
		curScript->OutputText(curScript->GetHelpString(iControlScript::_Exec));
		return true;
	}
	if(w == "print")
	{
		curScript->OutputText(curScript->GetHelpString(iControlScript::_Print));
		return true;
	}
	if(w=="create" || w=="delete")
	{
		curScript->OutputText(curScript->GetHelpString(iControlScript::_CreateDelete));
		return true;
	}
	if(w == "current")
	{
		curScript->OutputText(curScript->GetHelpString(iControlScript::_Current));
		return true;
	}
	if(w == "control")
	{
		curScript->OutputText(curScript->GetHelpString(iControlScript::_Control));
		return true;
	}
	if(w.IsEmpty())
	{
		curScript->OutputText(curScript->GetHelpString(iControlScript::_Root));
		return true;
	}

	return Print(self,arg);
}


bool iControlScript::HelpObject(iScript *self, const iString &arg)
{
	if(self->IsEmbedded()) return true;

	const iObjectType *tmp = iObjectTypeRegistry::FindType(arg);
	if(tmp != 0)
	{
		if(tmp->GetHelp() != 0) curScript->OutputText(tmp->GetHelp()->GetHTML()); else curScript->OutputText("<font color=#FF0000>No help is available for this item.</font>");
		return true;
	}
	else
	{
		self->GetErrorStatus()->Set("Invalid object name.");
		return false;
	}
}


bool iControlScript::HelpProperty(iScript *self, const iString &arg)
{
	if(self->IsEmbedded()) return true;

	const iObjectKey *tmp = iObjectKeyRegistry::FindKey(arg);
	if(tmp != 0)
	{
		if(tmp->GetHelp() != 0) curScript->OutputText(tmp->GetHelp()->GetHTML()); else curScript->OutputText("<font color=#FF0000>No help is available for this item.</font>");
		return true;
	}
	else
	{
		self->GetErrorStatus()->Set("Invalid object property.");
		return false;
	}
}


bool iControlScript::ListObjects(iScript *self)
{
	if(self->IsEmbedded()) return true;

	const int lmax = 10;
	int i;
	iString out, s;
	const iObjectType *ptr;

	for(i=0; i<lmax; i++) s += "$";

	out = "To get help for a specific object, use: <code><b>help object</b> <i>&lt;object-name&gt;</i></code><br>Available objects:<br>";
	out += "<i>Short form" + s.Part(0,lmax-10);
	out += "$$Long form</i><br><hr>";

	iObjectTypeRegistry::InitTraversal();
	while((ptr=iObjectTypeRegistry::GetNextType()) != 0)
	{
		out += "<br>$$" + ptr->ShortName() + s.Part(0,lmax-2-ptr->ShortName().Length()) + "$$" + ptr->FullName();
	}
	out += "</b><br>";
	out.Replace("$","&nbsp;");

	curScript->OutputText(out);

	return true;
}


bool iControlScript::ListProperties(iScript *self, const iString &arg)
{
	if(self->IsEmbedded()) return true;

	const int lmax1 = 12, lmax2 = 10;
	int i;
	iString out, v, s;
	const iObjectKey *ptr;

	const iObjectType *tmp = iObjectTypeRegistry::FindType(arg);
	if(tmp != 0)
	{
		for(i=0; i<lmax1+lmax2; i++) s += "$";

		out = "To get help for a specific property, use: <code><b>help property</b> &lt;<i>object-name</i>&gt;<b>:</b><i>property-name</i></code><br>Available properties:<br>";
		out += "$$<i>Type[Size]" + s.Part(0,lmax1-12);
		out += "$$Short form" + s.Part(0,lmax2-10);
		out += "$$Long form</i><br><hr>";

		iObjectKeyRegistry::InitTraversal(tmp);
		while((ptr=iObjectKeyRegistry::GetNextKey()) != 0) if(ptr->UnprefixedFullName().Part(0,9) != "___old___")
		{
			switch(ptr->Argument())
			{
			case iObjectKey::_OffsetInt:
			case iObjectKey::_Int:
				{
					v = "int";
					break;
				}
			case iObjectKey::_Bool:
				{
					v = "bool";
					break;
				}
			case iObjectKey::_Float:
				{
					v = "float";
					break;
				}
			case iObjectKey::_Double:
				{
					v = "double";
					break;
				}
			case iObjectKey::_Color:
				{
					v = "color";
					break;
				}
			case iObjectKey::_String:
				{
					v = "string";
					break;
				}
			case iObjectKey::_Any:
				{
					v = "any";
					break;
				}
			}
			out += "<b>" + s.Part(0,6-v.Length()) + v + "</b>[";
			if(ptr->Dimension() > 0)
			{
				v = iString::FromNumber(ptr->Dimension());
			}
			else
			{
				v = "*";
			}
			out += v + "]" + s.Part(0,lmax1-8-v.Length()) + "$$<b>" + ptr->UnprefixedShortName() + s.Part(0,lmax2-ptr->UnprefixedShortName().Length()) + "$$" + ptr->UnprefixedFullName() + "</b><br>";
		}
		out += "Use 0/1 for <code><b>false</b></code>/<code><b>true</b></code> for bool values and <code>int.int.int</code> for an RGB color (like 255.0.0 for red). Strings must not be included in quotes and contain slash characters (/). Use @@ if you need a slash character.<br>";
		out.Replace("$","&nbsp;");
		curScript->OutputText(out);
		return true;
	}
	else
	{
		self->GetErrorStatus()->Set("Invalid object name.");
		return false;
	}
}


//
//  This is whether the actual help text lives
//
const iString& iControlScript::GetHelpString(_HelpId id) const
{
	static const iString none = "No help available for this item";
	static const iString help[6] = 
	{
		//
		"Control Script understands the same flow control operators as the AnimatorScript. Type <code><b>help control</b></code> for the full description of the flow control. IFrIT consists of a collection of <b>objects</b>; each <b>object</b> is controlled by a set of <b>properties</b>.<br>"
			"Available commands:<br><blockquote>"
			"    <code><b>help</b> [<i>topic</i>]</code><br></blockquote>"
			"gives help on a specific topic. The short form of <code><b>help</b></code> is <code><b>.h</b></code>.<br><blockquote>"
			"    <code><b>list objects</b></code><br></blockquote>"
			"lists all available objects (short form <code><b>.lo</b></code>). Type <code><b>help object</b> &lt;<i>object-name</i>&gt;</code> (short form <code><b>.ho</b></code>) for help on a specific object.<br><blockquote>"
			"    <code><b>list properties</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
			"lists all properties for an object specified by <code>&lt;<i>object-name</i>&gt;</code> (short form <code><b>.lp</b></code>). Type <code><b>help property</b> &lt;<i>object-name</i>&gt;<b>:</b>&lt;<i>property-name</i>&gt;</code> (short form <code><b>.hp</b></code>) to get help on a specific property.<br><blockquote>"
			"    <code><b>exec</b> &lt;<i>request</i>&gt;</code><br></blockquote>"
			"executes a request (short form <code><b>.e</b></code>). Type <code><b>help exec</b></code> for more info.<br><blockquote>"
			"    <code><b>show</b> &lt;<i>object-name</i>&gt;</code><br>"
			"    <code><b>hide</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
			"shows/hides an object <code>&lt;<i>object-name</i>&gt;</code> in the scene (short forms <code><b>.s</b></code>/<code><b>.q</b></code>).<br><blockquote>"
			"    <code><b>print</b> &lt;<i>query</i>&gt;</code><br></blockquote>"
			"prints values of variables and properties (short form <code><b>.p</b></code>). Type <code><b>help print</b></code> for more info.<br><blockquote>"
			"    <code><b>create</b> &lt;<i>object-name</i>&gt;</code><br>"
			"    <code><b>delete</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
			"creates a new instance/deletes the current instance of an object <code>&lt;<i>object-name</i>&gt;</code> (short forms <code><b>.c</b></code> and <code><b>.d</b></code>). Type <code><b>help create</b></code> for more info.<br><blockquote>"
			"    <code><b>[set] current-window</b> &lt;<i>id</i>&gt;</code><br></blockquote>"
			"sets the viewization window specified by <code>&lt;<i>id</i>&gt;</code> as current (short form <code><b>.u</b></code>). Type <code><b>help current</b></code> for more info.<br><blockquote>"
			"    <code><b>save-state</b> [<i>file-name</i>]</code><br>"
			"    <code><b>load-state</b> [<i>file-name</i>]</code><br></blockquote>"
			"saves/loads the complete state of IFrIT from a file <code>&lt;<i>file-name</i>&gt;</code> (short forms <code><b>.ss</b></code> and <code><b>.ls</b></code>). If the file name is not specified, the default is used.<br><blockquote>"
			"    <code><b>animate</b> [<i>script-file-name</i>]</code><br></blockquote>"
			"animate the scene using the AnimatorScript from file <code>&lt;<i>script-file-name</i>&gt;</code> if it is specified, or using the current Animator settings otherwise (short form <code><b>.a</b></code>).<br><blockquote>"
			"    <code><b>render</b> &lt;<i>state</i>&gt;</code><br></blockquote>"
			"toggles whether the scene is rendered after each command (short form <code><b>.r</b></code>). <code>&lt;<i>state</i>&gt;</code> takes only two values: <code><b>on</b></code> or <code><b>off</b></code>.<br><blockquote>"
			"    <code><b>&lt;</b> &lt;<i>script-file-name</i>&gt;</code><br></blockquote>"
			"inserts the contents of <code>&lt;<i>script-file-name</i>&gt;</code> (which must contain valid ControlScript commands) into the current script as if they were typed one after another on the command line. If <code>&lt;<i>script-file-name</i>&gt;</code> starts with the plus sign (+), the name of the IFrIT base directory will be prepended to the name of the file.<br>",
			//
			"Format:<br><blockquote>"
			"    <code><b>exec</b> &lt;<i>request</i>&gt; [<i>request</i> ...]</code><br></blockquote>"
			"This command executes one or more control module requests for the current object. A request has the following form:<br><blockquote>"
			"    <code>&lt;<i>object-name</i>&gt;<b>:</b>&lt;<i>property-name</i>&gt;<b>/</b>&lt;<i>value</i>&gt;[<b>/</b><i>value</i>...]</code><br></blockquote>"
			"Here <code>&lt;<i>object-name</i>&gt;</code> is the string that encapsulates the object to which this request is addressed to (use <code><b>list objects</b></code> to get the list of all available objects), <code>&lt;<i>property-name</i>&gt;</code> is the property that serves the request (use <code><b>list properties</b> &lt;<i>object-name</i>&gt;</code> to get the list of all properties for the object <code>&lt;<i>object-name</i>&gt;</code>), and <code>&lt;<i>value</i>&gt;</code> is the value to be assigned to the property (can be an array). For example, the following command:<br><blockquote>"
			"    <code><b>exec Camera:ParallelProjection/0</b></code><br></blockquote>"
			"will set the projection in the viewization window to perspective. The same command can be typed in a short form as:<br><blockquote>"
			"    <code><b>.e c:pp/0</b></code><br></blockquote>"
			"where <code><b>.e</b></code> is the short form for <code><b>exec</b></code>. Two special forms of the <code><b>exec</b></code> command: <code><b>exec-all-objects</b></code> (short form <code><b>.eo</b></code>) and <code><b>exec-all-windows</b></code> (short form <code><b>.ew</b></code>) will execute the request for all instances of the object in the current viewization window and for all instances in all windows respectively.<br>",
			//
			"Format:<br><blockquote>"
			"    <code><b>print</b> &lt;<i>query</i>&gt;</code><br></blockquote>"
			"This command prints the value of <code>&lt;<i>query</i>&gt;</code>, which can be a script expression, a property in the form <code>&lt;<i>object-name</i>&gt;<b>:</b>&lt;<i>property-name</i>&gt;</code>, or the whole object <code>&lt;<i>object-name</i>&gt;<b>:</b></code> (the semi-colon at the end is required, to separate an object from a regular variable). Use <code><b>list objects</b></code> to get the list of all available objects and <code><b>list properties</b> &lt;<i>object-name</i>&gt;</code> to get the list of all available properties for a given object <code>&lt;<i>object-name</i>&gt;</code>.<br>",
			//
			"Format:<br><blockquote>"
			"    <code><b>create</b> &lt;<i>object-name</i>&gt;</code><br>"
			"    <code><b>delete</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
			"These commands create or remove a new instance of a viewization object <code>&lt;<i>object-name</i>&gt;</code>. The valid values for <code>&lt;<i>object-name</i>&gt;</code> are <code><b>Surface</b></code>, <code><b>CrossSection</b></code>, <code><b>ParticlesGroup</b></code>, and <code><b>View</b></code> (other objects cannot have multiple instances). For view modules, an additional option <code><b>/copy</b></code> or <code><b>/clone</b></code> is permitted, in order to create a copy or a clone of the current view module.<br>",
			//
			"Format:<br><blockquote>"
			"    <code><b>[set] current-window =</b> &lt;<i>id</>&gt;</code><br></blockquote>"
			"This command sets the viewization window #<code>&lt;<i>id</i>&gt;</code> as current. Here <code>&lt;<i>id</>&gt;</code> is an integer number from 1 to the number of windows available. For example, the following 2 commands create a new viewization window (module) as a clone of the furst one, and make the second window current:<br><blockquote>"
			"    <code><b>create View /clone</b></code><br>"
			"    <code><b>current-window = 2</b></code><br></blockquote>",
			//
			"Variable declaration:<br><blockquote>"
			"    <code><b>var</b> &lt;<i>type</i>&gt;[<b>[</b><i>dim</i><b>]</b>] &lt;<i>name1</i>&gt; [<b>,</b> <i>name2</i> ...]</code><br></blockquote>"
			"declares a variable(s) of type <code>&lt;<i>type</i>&gt;</code> and name(s) <code>&lt;<i>name1</i>&gt;</code> (<code>&lt;<i>name2</i>&gt;</code> ...) with the optional dimension <code>&lt;<i>dim</i>&gt;</code>. More than one variable of the same type can be declared in a single statement. <code>&lt;<i>type<i/>&gt;</code> is either <code><b>int</b></code>, <code><b>real</b></code>, or <code><b>bool</b></code>. For example, the following statement declares two integer arrays <code><i>p</i></code> and <code><i>q</i></code> with 5 elements each:<br><blockquote>"
			"    <code><b>var int[5] p, q</b></code><br></blockquote>"
			"Declared variables can be assigned values and can be used in expressions:<br><blockquote>"
			"    <code><b>q[1] = 3</b></code><br>"
			"    <code><b>p += (1,2,3,4,5)</b></code><br>"
			"    <code><b>q[2] *= q[1]*sin(q[3]^2)</b></code><br></blockquote><br>"
			"Loop statements:<br><blockquote>"
			"    <code><b>loop</b> &lt;<i>count</i>&gt;</code><br>"
			"    <code>...</code><br>"
			"    <code><b>end</b></code><br></blockquote>"
			"or<br><blockquote>"
			"    <code><b>for</b> &lt;<i>var</i>&gt; <b>to</b> &lt;<i>count</i>&gt;</code><br>"
			"    <code>...</code><br>"
			"    <code><b>end</b></code><br></blockquote>"
			"repeat the statements within the body of the loop <code>&lt;<i>count</i>&gt;</code> times. The second form also uses the variable <code>&lt;<i>var</i>&gt;</code> (which must be declared) as a loop index, taking consequitive values from 1 to <code>&lt;<i>count</i>&gt;</code>.<br><br>"
			"Conditional statement:<br><blockquote>"
			"    <code><b>if</b> &lt;<i>boolean-expression</i>&gt; <b>then</b></code><br>"
			"    <code>...</code><br>"
			"    <code><b>else</b></code><br>"
			"    <code>...</code><br>"
			"    <code><b>endif</b></code><br></blockquote>"
			"behaves as expected. Use usual C-style comparison operators to compare numbers: <code><b>&lt;</b></code> <code><b>&gt;</b></code> <code><b>&lt;=</b></code> <code><b>&gt;=</b></code> <code><b>&&</b></code> <code><b>||</b></code> <code><b>!=</b></code>.<br><br>"
			"The text of the AnimatorScript can be embedded into the body of the ControlScript using the embedding statement:<br><blockquote>"
			"    <code><b>embed animator-script</b></code><br>"
			"    <code><b>&gt; </b><i>first-animator-script-statement</i></code><br>"
			"    <code><b>&gt; </b><i>second-animator-script-statement</i></code><br>"
			"    <code><b>&gt; </b>...</code><br>"
			"    <code><b>&gt; </b><i>last-animator-script-statement</i></code><br></blockquote><br>"
	};

	if(id>=0 && id<8) return help[id]; else return none;
}

