#include <string.h>
#include <stdlib.h>

#include "gm-string.h"
#include "gm-debug.h"

gboolean
gm_string_to_int(const gchar *str, int *result) {
	*result = 0;
	
	if (str == NULL || *str == '\0') {
		return FALSE;
	}
	
	*result = atoi(str);
	return TRUE;
}

gchar *
gm_string_catn(gchar *str, gchar *add, guint n) {
	guint relen;
	gchar *newstr;

	if (str == NULL) {
		relen = 0;
	} else {
		relen = strlen(str);
	}

	if ((newstr = (gchar *) realloc(str, (relen + n + 1) * sizeof(gchar))) == NULL) {
		gm_debug_msg(DEBUG_DEFAULT, "mystring_catn: REALLOC FAILED!");
		return str;
	} else {
		if (relen == 0) {
			newstr[0] = '\0';
		}
		
		strncat(newstr, add, n);
		return newstr;
	}
}

gchar *
gm_string_cat(gchar *str, gchar *add) {
	return gm_string_catn(str, add, strlen(add));
}

gchar *
gm_string_trim(const gchar *str) {
  gchar *newstr = NULL;
  const gchar *r, *l;

  if (str == NULL) {
    return NULL;
  }

  r = str + strlen(str);
  l = str;

  while (*l == ' ' || *r == ' ') {
    if (l == r) {
      break;
    }

    if (*l == ' ') {
      l++;
    }

    if (l == r) {
      break;
    }

    if (*r == ' ') {
      r--;
    }
  }

  if (l == r) {
    return g_strdup("");
  } else {
    newstr = g_strndup(l, r - l);
    return newstr;
  }
}

gchar *
gm_string_remove_char(gchar *str, gchar rem) {
	gchar *start = str;
	gchar *ptr = str;
	gboolean changed = FALSE;
	gunichar c;
		
	while (*str != '\0') {
		c = g_utf8_get_char(str);
		
		if (c != (gunichar)rem) {
			if (changed) {
				ptr += g_unichar_to_utf8(c, ptr);
			} else {
				ptr = g_utf8_next_char(ptr);
			}
		} else {
			changed = TRUE;
		}
		
		str = g_utf8_next_char(str);
	}
	
	*ptr = '\0';
	
	return start;
}

void
gm_string_skip_space(gchar const **ptr) {
	while (**ptr != '\0' && g_unichar_isspace(g_utf8_get_char(*ptr))) {
		*ptr = g_utf8_next_char(*ptr);
	}
}

void
gm_string_skip_nonspace(gchar const **ptr) {
	while (**ptr != '\0' && !g_unichar_isspace(g_utf8_get_char(*ptr))) {
		*ptr = g_utf8_next_char(*ptr);
	}
}

void
gm_string_skip_till(gchar const **ptr, gchar const *find) {
	gchar const *fptr;
	gunichar check, c;
	
	while (**ptr != '\0') {
		check = g_utf8_get_char(*ptr);
		fptr = find;
		
		while (*fptr != '\0') {
			c = g_utf8_get_char(fptr);
			
			if (check == c) {
				return;
			}
			
			fptr = g_utf8_next_char(fptr);
		}
		
		*ptr = g_utf8_next_char(*ptr);
	}
}

gchar *
gm_string_escape(gchar const *line) {
	GString *new_line;
	gchar *ptr;
	gunichar c;

	if (*line == '\0') {
		return g_strdup("");
	}

	if (g_utf8_strchr(line, '"', -1) || g_utf8_strchr(line, '\\', -1)) {
		// We need at least strlen(line)
		new_line = g_string_sized_new(strlen(line));
		
		while (*line != '\0') {
			c = g_utf8_get_char(line);
			
			if (c == '"' || c == '\\') {
				g_string_append_c(new_line, '\\');
			}
			
			g_string_append_unichar(new_line, c);
	    	line = g_utf8_next_char(line);
		}
		
		ptr = new_line->str;
		g_string_free(new_line, FALSE);
		
		return ptr;
	} else {
		return g_strdup(line);
	}
}

GList *
gm_string_split_add_line(gchar *line, GList *lines, GList **lastptr) {
	if (!lines) {
		lines = g_list_append(NULL, line);
		*lastptr = lines;
	} else {
		*lastptr = g_list_append(*lastptr, line);
		*lastptr = (*lastptr)->next;
	}
	
	return lines;
}

GList *
gm_string_split(gchar const *text) {
	gchar *fptr, *line;
	GList *lastptr;
	GList *lines = NULL;
	
	if (!text) {
		return NULL;
	}
	
	while ((fptr = g_utf8_strchr(text, -1, '\n'))) {
		line = g_strndup(text, fptr - text);
		gm_string_remove_char(line, '\r');
		
		lines = gm_string_split_add_line(line, lines, &lastptr);
		text = g_utf8_next_char(fptr);
	}
	
	if (*text != '\0') {
		line = g_strdup(text);
		gm_string_remove_char(line, '\r');
		
		lines = gm_string_split_add_line(line, lines, &lastptr);
	}
	
	return lines;
}
