use std::{
    fs::OpenOptions,
    io::Read,
    path::{Path, PathBuf},
    process::Command,
    str::FromStr,
};

use log::{debug, error, info, warn};
use pci_device::GfxVendor;

use crate::{error::GfxError, pci_device::GfxMode, special_asus::*};

/// The configuration for graphics. This should be saved and loaded on boot.
pub mod config;
mod config_old;
/// Control functions for setting graphics.
pub mod controller;
/// Error: 404
pub mod error;
/// Special-case functions for check/read/write of key functions on unique laptops
/// such as the G-Sync mode available on some ASUS ROG laptops
pub mod special_asus;

/// Defined DBUS Interface for supergfxctl
pub mod zbus_iface;
/// Defined DBUS Proxy for supergfxctl
pub mod zbus_proxy;

/// System interface helpers.
pub mod pci_device;

/// Systemd helpers
pub mod systemd;

/// The actual actions that supergfx uses for each step
pub mod actions;

#[cfg(test)]
mod tests;

/// Helper to expose the current crate version to external code
pub const VERSION: &str = env!("CARGO_PKG_VERSION");
/// Generic path that is used to save the daemon config state
pub const CONFIG_PATH: &str = "/etc/supergfxd.conf";
/// Destination name to be used in the daemon when setting up DBUS connection
pub const DBUS_DEST_NAME: &str = "org.supergfxctl.Daemon";
/// Generic icd-profile (vulkan)
pub const CONFIG_NVIDIA_VKICD: &str = "/usr/share/vulkan/icd.d/nvidia_icd.json";
/// Interface path name. Should be common across daemon and client.
pub const DBUS_IFACE_PATH: &str = "/org/supergfxctl/Gfx";

pub const KERNEL_CMDLINE: &str = "/proc/cmdline";

const SLOTS: &str = "/sys/bus/pci/slots";

const NVIDIA_DRIVERS: [&str; 4] = ["nvidia_drm", "nvidia_modeset", "nvidia_uvm", "nvidia"];

const VFIO_DRIVERS: [&str; 6] = [
    "vfio_pci",
    "vfio_pci_core",
    "vfio_iommu_type1",
    "vfio_virqfd",
    "vfio_mdev",
    "vfio",
];

const DISPLAY_MANAGER: &str = "display-manager.service";

const MODPROBE_PATH: &str = "/etc/modprobe.d/supergfxd.conf";

static MODPROBE_NVIDIA_BASE: &[u8] = br#"# Automatically generated by supergfxd
blacklist nouveau
alias nouveau off
"#;

static MODPROBE_NVIDIA_DRM_MODESET_ON: &[u8] = br#"
options nvidia-drm modeset=1
"#;

// static MODPROBE_NVIDIA_DRM_MODESET_OFF: &[u8] = br#"
// options nvidia-drm modeset=0
// "#;

static MODPROBE_INTEGRATED: &[u8] = br#"# Automatically generated by supergfxd
blacklist nouveau
blacklist nvidia_drm
blacklist nvidia_uvm
blacklist nvidia_modeset
blacklist nvidia
"#;

static MODPROBE_VFIO: &[u8] = br#"options vfio-pci ids="#;

#[derive(Debug, Clone, Copy)]
pub enum DriverAction {
    Remove,
    Load,
}

impl From<DriverAction> for &str {
    fn from(a: DriverAction) -> Self {
        match a {
            DriverAction::Remove => "rmmod",
            DriverAction::Load => "modprobe",
        }
    }
}

/// Basic check for support. If `()` returned everything is kosher.
fn mode_support_check(mode: &GfxMode) -> Result<(), GfxError> {
    if matches!(mode, GfxMode::AsusEgpu) && !asus_egpu_enable_exists() {
        let text = "Egpu mode requested when either the laptop doesn't support it or the kernel is not recent enough".to_string();
        return Err(GfxError::NotSupported(text));
    }
    Ok(())
}

/// Add or remove driver modules
fn do_driver_action(driver: &str, action: DriverAction) -> Result<(), GfxError> {
    let mut cmd = Command::new(<&str>::from(action));
    cmd.arg(driver);

    let mut count = 0;
    const MAX_TRIES: i32 = 6;
    loop {
        if count > MAX_TRIES {
            let msg = format!(
                "{} {} failed for unknown reason",
                <&str>::from(action),
                driver
            );
            error!("{}", msg);
            break; //Err(GfxError::Modprobe(msg));
        }

        let output = cmd
            .output()
            .map_err(|err| GfxError::Command(format!("{:?}", cmd), err))?;
        if !output.status.success() {
            if output
                .stderr
                .ends_with("is not currently loaded\n".as_bytes())
            {
                debug!(
                    "Driver {driver} was not loaded, skipping {}",
                    <&str>::from(action)
                );
                break;
            }
            if output.stderr.ends_with("is builtin.\n".as_bytes()) {
                return Err(GfxError::VfioBuiltin);
            }
            if output.stderr.ends_with("Permission denied\n".as_bytes()) {
                warn!(
                    "{} {} failed: {:?}",
                    <&str>::from(action),
                    driver,
                    String::from_utf8_lossy(&output.stderr)
                );
                warn!("It may be safe to ignore the above error, run `lsmod |grep {}` to confirm modules loaded", driver);
                break;
            }
            if String::from_utf8_lossy(&output.stderr)
                .contains(&format!("Module {} not found", driver))
            {
                return Err(GfxError::MissingModule(driver.into()));
            }
            if count >= MAX_TRIES {
                let msg = format!(
                    "{} {} failed: {:?}",
                    <&str>::from(action),
                    driver,
                    String::from_utf8_lossy(&output.stderr)
                );
                return Err(GfxError::Modprobe(msg));
            }
        } else if output.status.success() {
            debug!("Did {} for driver {driver}", <&str>::from(action));
            break;
        }

        count += 1;
        std::thread::sleep(std::time::Duration::from_millis(50));
    }
    Ok(())
}

pub fn toggle_nvidia_powerd(run: bool, vendor: GfxVendor) -> Result<(), GfxError> {
    if vendor == GfxVendor::Nvidia {
        let mut cmd = Command::new("systemctl");
        if run {
            cmd.arg("start");
        } else {
            cmd.arg("stop");
        }
        cmd.arg("nvidia-powerd.service");

        let status = cmd.status()?;
        if !status.success() {
            warn!("{run} nvidia-powerd.service failed: {:?}", status.code());
        }
        debug!("Did {:?}", cmd.get_args());
    }
    Ok(())
}

pub fn kill_nvidia_lsof() -> Result<(), GfxError> {
    if !PathBuf::from("/dev/nvidia0").exists() {
        return Ok(());
    }

    if !PathBuf::from("/usr/bin/lsof").exists() {
        warn!("The lsof util is missing from your system, please ensure it is available so processes hogging Nvidia can be nuked");
        return Ok(());
    }

    let mut cmd = Command::new("lsof");
    cmd.arg("/dev/nvidia0");

    let output = cmd
        .output()
        .map_err(|err| GfxError::Command(format!("{:?}", cmd), err))?;

    let st = String::from_utf8_lossy(&output.stdout);

    for line in st.lines() {
        let mut split = line.split_whitespace();
        if let Some(c) = split.next() {
            if let Some(pid) = split.next() {
                if let Ok(pid) = pid.parse::<u32>() {
                    warn!("pid {pid} ({c}) is holding /dev/nvidia0. Killing");
                    let mut cmd = Command::new("kill");
                    cmd.arg("-9");
                    cmd.arg(format!("{pid}"));
                    let status = cmd
                        .status()
                        .map_err(|err| GfxError::Command(format!("{:?}", cmd), err))?;
                    if !status.success() {
                        warn!("Killing pid {pid} failed");
                    }
                }
            }
        }
    }

    Ok(())
}

pub fn get_kernel_cmdline_mode() -> Result<Option<GfxMode>, GfxError> {
    let path = Path::new(KERNEL_CMDLINE);
    let mut file = OpenOptions::new()
        .read(true)
        .open(path)
        .map_err(|err| GfxError::Path(KERNEL_CMDLINE.to_string(), err))?;
    let mut buf = String::new();
    file.read_to_string(&mut buf)?;

    // No need to be fast here, just check and go
    for cmd in buf.split(' ') {
        if cmd.contains("supergfxd.mode=") {
            let mode = cmd.trim_start_matches("supergfxd.mode=");
            let mode = GfxMode::from_str(mode)?;
            return Ok(Some(mode));
        }
    }

    info!("supergfxd.mode not set, ignoring");
    Ok(None)
}

pub fn get_kernel_cmdline_nvidia_modeset() -> Result<Option<bool>, GfxError> {
    let path = Path::new(KERNEL_CMDLINE);
    let mut file = OpenOptions::new()
        .read(true)
        .open(path)
        .map_err(|err| GfxError::Path(KERNEL_CMDLINE.to_string(), err))?;
    let mut buf = String::new();
    file.read_to_string(&mut buf)?;

    // No need to be fast here, just check and go
    for cmd in buf.split(' ') {
        if cmd.contains("nvidia-drm.modeset=") {
            let mode = cmd.trim_start_matches("nvidia-drm.modeset=");
            let mode = mode == "1";
            return Ok(Some(mode));
        }
    }

    info!("nvidia-drm.modeset not set, ignoring");
    Ok(None)
}

pub fn find_slot_power(address: &str) -> Result<PathBuf, GfxError> {
    let mut buf = Vec::new();
    let path = PathBuf::from_str(SLOTS).unwrap();
    for path in path.read_dir()? {
        let path = path.unwrap().path();

        let mut address_path = path.to_path_buf();
        address_path.push("address");

        let mut file = OpenOptions::new().read(true).open(&address_path)?;
        file.read_to_end(&mut buf)?;

        if address.contains(String::from_utf8_lossy(&buf).trim_end()) {
            address_path.pop();
            address_path.push("power");
            info!("Found hotplug power slot at {:?}", address_path);
            return Ok(address_path);
        }
        buf.clear();
    }
    Err(GfxError::DgpuNotFound)
}
